#-------------------------------------------------------------------
#  JsonResponseDesc.py
#
#  The JsonResponseDesc class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the JsonResponseDesc class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonFieldDesc
from ai.axe.classJson.jsonTypes import JsonObj
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.util import StringUtil
from typing import Dict
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonResponseDesc:
  '''Desciption of objects to be returns in a JSON response.

  This object takes a list of ClassJsonField objects.  Each
  describes a field that will be included in the response.

  For example, if you specified your response like so:

  JsonResponseDesc((ClassJsonField('exists', 'bool'),
                    ClassJsonField('foo', 'obj', objClass='Foo'))

  then your action function would return a dictionary of objects
  to be turned into a JSON string.

  return { 'exists' : True, 'foo' : Foo(data1, data2) };

  and the caller would receive the string:

     {
      'exists' : true,
      'foo' :
      {
        <foo data here >
      }
    }
  '''

  #-----------------------------------------------------------------
  def __init__(self, fields: List[ClassJsonField]) -> None:
    '''Creates a new JsonResponseDesc.
    '''

    # The field that will be present in the response.
    self.fields: List[ClassJsonFieldDesc] = [x.toDesc() for x in fields]

    # The field that will be present in the response.
    self.fieldsRaw: List[ClassJsonField] = fields

    # The @ClassJsonClass-decorated class to use to convert to JSON.
    self.jsonClass: Optional[type] = None

  #-----------------------------------------------------------------
  def getJsonClass(self) -> type:
    '''Returns the artifical class created for converting objects to JSON.
    '''

    assert self.jsonClass is not None

    return self.jsonClass

  #-----------------------------------------------------------------
  def setJsonClassName(self, className: str) -> None:
    '''Sets the '_class' name to include in the JSON.
    '''

    assert self.jsonClass is None

    self.createJsonClass(className)

  #-----------------------------------------------------------------
  def findAllClasses(self) -> List[JsonObj]:
    '''Returns all ClassJsonClasses in this class or children.
    '''

    jsonObjs: List[JsonObj] = []

    for classJsonField in self.fields:

      jsonType = classJsonField.jsonType
      for childObj in jsonType.childJsonObjs():
        assert isinstance(childObj, JsonObj)
        jsonObjs.append(childObj)

    # Add in any subclasses.

    for jsonObj in jsonObjs[:]:
      jsonObjs.extend(JsonObjRegistry.findAllClasses(jsonObj))

    return jsonObjs

  #-----------------------------------------------------------------
  def createObject(self, paramDict: Dict[str, object]) -> object:
    '''Creates a @ClassJsonClass-decorated class object from the params.

    This should be used to create an object that can be converted
    to JSON using a ClassJsonEncoder.

    @param paramDict The values to set on the created object.

    @return An object of type of a @ClassJsonClass-decorated class.
    '''

    assert self.jsonClass is not None

    return self.jsonClass(paramDict)

  #-----------------------------------------------------------------
  def createJsonClass(self, className: str) -> None:
    '''Creates a @ClassJsonClass-decorated class to use to encode values.
    '''

    assert self.jsonClass is None

    # The @ClassJsonClass decorator uses the class name, so we must
    # create the class, set its name, then decorate it manually.

    # Create the class with a temporary name.

    class NewClass:
      '''Generated class for WebAction return values.
      '''

      def __init__(self, paramDict):

        for key in paramDict:
          setattr(self, key, paramDict[key])

    # Set the correct class name.
    NewClass.__name__ = str('ram.webapi.classJson.gen.' + className)

    # Decorate the class.
    decorator = ClassJsonClass(self.fieldsRaw, encodeOnly=True)
    decorator(NewClass)


    self.jsonClass = NewClass

  #-----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Format this object into a string.
    '''

    attrs = ['fields']
    return StringUtil.formatRepr(self, attrs)
