#-------------------------------------------------------------------
#  DocParser.py
#
#  The DocParser class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the DocParser class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .DocComment import DocComment
from collections import OrderedDict
from typing import Dict
from typing import List
from typing import Tuple
import string
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class DocParser:
  '''Parses python doc strings.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new DocParser.
    '''

    pass

  #-----------------------------------------------------------------
  def parse(self, text: str) -> DocComment:
    '''Returns a DocComment object for the specified text.

    @param text The string to parse.

    @return an equivalent DocComment.
    '''

    text = self.trimWhite(text)

    lines = text.split('\n')

    shortDesc, lines = self.nextParagraph(lines, findTag=False)

    longDesc: List[str] = []

    while True:
      paragraph, lines = self.nextParagraph(lines, findTag=False)
      if not paragraph:
        break
      longDesc.append(paragraph)

    tagList = []

    while True:
      paragraph, lines = self.nextParagraph(lines, findTag=True)
      if not paragraph:
        break
      tagList.append(paragraph)

    params: Dict[str, str] = OrderedDict()
    returnDesc: str = ''

    for tagText in tagList:
      assert tagText.startswith('@')
      tagName = tagText.split(' ', 1)[0]
      tagText = tagText[len(tagName):].strip()

      if tagName in ('@param', '@webParam'):
        paramName = tagText.split(' ', 1)[0]
        tagText = tagText[len(paramName):].strip()
        params[paramName] = tagText

      elif tagName == '@return':
        returnDesc = tagText

    return DocComment(shortDesc, longDesc, params, returnDesc)

  #-----------------------------------------------------------------
  def nextParagraph(self,
                    lines: List[str],
                    findTag: bool) -> Tuple[str, List[str]]:
    '''Returns the next paragraph contained within the list of lines.

    @param lines A list of strings.
    @param findTag If true, the next tag will be found
                   If false, the next paragraph will be found.

    @return a (string, lines) tuple.  The next paragraph (or '' if no
            paragraph was found) and the list of lines with lines put
            into the paragraph removed.
    '''

    allLines = lines[:]
    unusedLines = []

    # A paragraph begins when a non-blank, non-tag line is found.

    while allLines:
      if allLines[0] and (not findTag and self.isTagLine(allLines[0])):
        # The first line we saw was a tag.
        # That means we're past the description paragraphs.
        return '', allLines

      if not allLines[0] or (findTag and not self.isTagLine(allLines[0])):
        unusedLines.append(allLines.pop(0))
      else:
        break

    # A paragraph ends when:
    #  - the end of the text is reached.
    #  - a blank line is found.
    #  - a tag is found.

    endIdx = 1
    while len(allLines) > endIdx:
      if not allLines[endIdx] or self.isTagLine(allLines[endIdx]):
        break
      else:
        endIdx += 1

    unusedLines.extend(allLines[endIdx:])
    allLines = allLines[:endIdx]

    trimmedLines = [line.strip() for line in allLines]
    paragraph = ' '.join(trimmedLines)

    return paragraph, unusedLines

  #-----------------------------------------------------------------
  def isTagLine(self, line: str) -> bool:
    '''Returns true if the spacified line is a tag.

    @param line The line to test.

    @return True if the spacified line is a tag.
    '''

    return line[0] == '@'

  #-----------------------------------------------------------------
  def trimWhite(self, text: str) -> str:
    '''Returns the specified text with whitespace removed.

    @param text The string to trim.

    @return the trimmed string.
    '''

    lines = text.strip().split('\n')

    minLen = 999
    for i, line in enumerate(lines[1:]):
      hasNonWhite = False
      numWhite = 0
      for ch in line:
        if ch in string.whitespace:
          numWhite += 1
        else:
          hasNonWhite = True
          break
      if hasNonWhite:
        minLen = min(minLen, numWhite)
      else:
        # Remove stray whitespace.
        lines[i + 1] = ''

    # Trim the rest of the lines.
    for i, line in enumerate(lines[1:]):
      if line:
        lines[i + 1] = line[minLen:]

    text = '\n'.join(lines)
    return text
