#-------------------------------------------------------------------
#  DetailDocWriter.py
#
#  The DetailDocWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the DetailDocWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonSchemaWriter
from ai.axe.web.api.BinaryResponse import BinaryResponse
from ai.axe.web.api.DownloadResponse import DownloadResponse
from .JsonResponseDesc import JsonResponseDesc
from .WebVerbDesc import WebVerbDesc
from typing import Dict
from typing import List
from werkzeug.wrappers import Response
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class DetailDocWriter:
  '''Generates html for detailed documentation page
  '''

  introHtmlText = """<!DOCTYPE html>
<html>
  <head>

    <style>
      /* CSS here */
    </style>

  </head>
  <body>
"""


  outroHtmlText = '''
    </table>
  </body>
</html>'''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new DetailDocWriter.
    '''

    pass

  #-----------------------------------------------------------------
  def formatHTMLVerbDict(self, verbDict: Dict[str, List[WebVerbDesc]]) -> str:
    '''Formats the verb dict into an HTML page.
    '''

    text = "<table>"
    linkText = "<a href=\"#"
    writer = ClassJsonSchemaWriter()
    headerText = "<h1><a id=\"top\">Gmc Web API Docs</a></h1>"

    nounLinks = headerText
    for noun, verbList in verbDict.items():
      nounLinks += linkText + noun + "\">" + noun + "</a><br>"
      text += self.newLine("<a id=" + noun + "><h2>" + noun + "</h2></a>")
      for verbDesc in verbList:
        docComment = verbDesc.docComment
        verbStr = verbDesc.verb
        if verbDesc.flavor:
          verbStr += '&#47;' + verbDesc.flavor
        text += self.newLine("<h3>" + verbStr + "</h3>")
        text += self.newLine(docComment.shortDesc, 1)

        text += self.newLine("Params:", 1)
        for paramDesc, paramDoc in zip(verbDesc.requestParams,
                                       docComment.params.values()):

          text += self.newLine(paramDesc.effectiveUrlName() + '</td><td>' +
                               paramDesc.paramType.label(paramDesc.objClass) +
                               '</td><td>' +  paramDoc + '</td>', 1)

        paramObjClasses: List[type] = []
        for paramDesc in verbDesc.requestParams:
          if paramDesc.objClass and paramDesc.objClass not in paramObjClasses:
            paramObjClasses.append(paramDesc.objClass)

        text += self.newLine('Param clases:', 1)
        for objClass in paramObjClasses:
          text += self.newLine(writer.writeClass(objClass, 3, True), 1)

        text += self.newLine('Response:', 1)
        if isinstance(verbDesc.responseClass, str):
          text += self.newLine(verbDesc.responseClass, 1)
        elif verbDesc.responseClass in (BinaryResponse, DownloadResponse):
          assert isinstance(verbDesc.responseClass, type)
          text += self.newLine(verbDesc.responseClass.__name__, 1)
        elif verbDesc.responseClass == Response:
          text += self.newLine('Raw HTTP response', 1)
        elif isinstance(verbDesc.responseClass, JsonResponseDesc):
          fields = verbDesc.responseClass.fields
          text += "<tr>"
          rawResponse = writer.writeFields(None, fields, 3, True)
          text += "</tr>"
          text += rawResponse
        else:
          assert isinstance(verbDesc.responseClass, type)
          text += writer.writeClass(verbDesc.responseClass,
                                    3,
                                    formatForHtml=True)
      text += (self.newLine(linkText +"top\">Back to Top</a>") +
                                     '</table><table>')
    text += self.outroHtmlText

    return self.introHtmlText + nounLinks + text

  #-----------------------------------------------------------------
  def indent(self, num: int = 1) -> str:
    '''Returns an indent string equal to the size of num

    @param num number of indents to do
    '''

    return "<td></td>" * num

  #-----------------------------------------------------------------
  def newLine(self, text: str, indents: int = 0) -> str:
    '''Returns a new line with the passed in text in the middle

    @param text some text for a new line
    '''

    indentStr = self.indent(indents)

    return "<tr>" + indentStr + "<td>" + text + "</td></tr>"

  #-----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
