#-------------------------------------------------------------------
#  WebWorkerRunner.py
#
#  The WebWorkerRunner class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebWorkerRunner class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .WebWorkerContext import WebWorkerContext
from ai.axe.db.DatabaseMgr import DatabaseMgr
from datetime import datetime
from multiprocessing import Process
from typing import Any
from typing import Callable
from typing import List
from typing import Optional
import traceback
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WebWorkerRunner:
  '''Runs a single action.
  '''

  #-----------------------------------------------------------------
  def __init__(self, workerContext: WebWorkerContext) -> None:
    '''Creates a new WebWorkerRunner.
    '''

    # The action which this worker was started by.
    self.workerContext: WebWorkerContext = workerContext

    # The name of this action.
    self.name: Optional[str] = None

    # The function that this runner will run.
    self.workerFunction: Optional[Callable[..., Any]] = None

    # Arguments that should be passed to the worker function.
    self.workerFunctionArgs: Optional[List[str]] = None

    # When the woker function began.
    self.beginTime: Optional[datetime] = None

  #-----------------------------------------------------------------
  def init(self,
           name: str,
           workerFunction: Callable[..., Any],
           workerFunctionArgs: List[str]) -> None:
    '''Initializes this runner to run the specified function.
    '''

    self.name = name
    self.workerFunction = workerFunction
    self.workerFunctionArgs = workerFunctionArgs

  #-----------------------------------------------------------------
  def start(self) -> None:
    '''Starts the worker function running in a new process.
    '''

    assert self.runWorkerFunction is not None

    process = Process(target=self.runWorkerFunction)
    process.start()

  #-----------------------------------------------------------------
  def runWorkerFunction(self) -> None:
    '''Runs the worker function.
    '''

    assert self.name is not None
    assert self.workerFunction is not None
    assert self.workerFunctionArgs is not None

    self.beginTime = datetime.utcnow()

    # Set up this process' actionProcessor to write a web worker name.
    self.workerContext.setWebWorkerName(self.name)

    databaseMgr: DatabaseMgr = self.workerContext.getDatabaseMgr()

    # Clear out the connection pool in this child process, so this process
    # and the parent process don't try to connect to the DB on the same ports.
    databaseMgr.recreateSessionMaker()

    # The DB session for this request.
    session = databaseMgr.create()

    allArgs = (session,) + tuple(self.workerFunctionArgs)

    try:

      self.workerFunction(*allArgs)

      session.commit()
      self.workerContext.doPostSuccess()

    # pylint: disable = W0703
    except Exception:

      self.workerContext.doPostFailure(traceback.format_exc())
      session.rollback()

    finally:

      session.close()
