#-------------------------------------------------------------------
#  RunServer.py
#
#  The RunServer module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The 'runServer' script implementation.

To use this in a dev environment, make a script called 'runServer'
whose implementation is:

  MyAppSetup.register(MyAppSetup())
  RunServer.execute()
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import RequestProcessor
from ai.axe.build import BuildIdFile
from ai.axe.build import BuildSetup
from ai.axe.build import VersionFile
from ai.axe.web.app import AppSetup
from ai.axe.web.build import Typescript
from flask import Flask
from typing import List
import os.path
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
# Usage string to display to the user.
#
# FRONT_END_OPTIONS and FRONT_END_DETAIL are to be included only if
# the app has a browser-based front end.
usage = '''runServer FRONT_END_OPTIONS

Runs the RAM developement web server.  This server is much slower
than a production server and uses HTTP instead of HTTPS.
FRONT_END_DETAIL'''

usageFrontEndOptions = ' [-debug|-release] [-test]'

usageFrontEndDetail = '''
If '-test' is provided, then the Typescript unit tests will be compiled
into the app.  To run the tests, start the app with "?unitTest=testName"
or simply "?unitTest" (to run all tests) appended to the URL in the
browser location input box.

If '-debug' is provided (the default) seperate source files are served,
which show more debug information in the console.

If '-release' is specified, files are combined into one huge .js and .css
as on the production server.
'''

#-------------------------------------------------------------------
def getUsage() -> str:
  '''Returns the usage string.
  '''

  appSetup = AppSetup.get()
  hasFrontEnd = appSetup.hasFrontEnd

  frontEndOptions = usageFrontEndOptions if hasFrontEnd else ""
  frontEndDetail = usageFrontEndDetail if hasFrontEnd else ""

  retUsage = usage

  retUsage = retUsage.replace(" FRONT_END_OPTIONS", frontEndOptions)
  retUsage = retUsage.replace("FRONT_END_DETAIL", frontEndDetail)

  return retUsage

#-------------------------------------------------------------------
def execute() -> None:
  '''Executes this command.

  @param port The port to run the server on.
  '''

  appSetup: AppSetup = AppSetup.get()
  hasFrontEnd: bool = appSetup.hasFrontEnd()

  appName: str = appSetup.appName()
  appNameAllCaps: str = appSetup.appNameAllCaps()

  home: str = os.environ[appNameAllCaps + '_HOME']
  webRoot: str = os.path.join(home, "build", "webapp")

  port: int = BuildSetup.get().localDevServerPort()

  # Process the command line arguments.

  args: List[str] = sys.argv[1:]

  if '-h' in args or '--help' in args:
    print(getUsage())
    sys.exit(0)

  isDebug = True
  includeUnitTests = False

  for arg in args:
    if hasFrontEnd and arg == '-debug':
      isDebug = True
    elif hasFrontEnd and arg == '-release':
      isDebug = False
    elif hasFrontEnd and arg == '-test':
      includeUnitTests = True
    else:
      print(('Error. Unknown argument:', arg))
      sys.exit(1)

  # Start the server.

  # The server will restart if any of these files change.
  restartFileNames: List[str] = []

  processor = RequestProcessor(noisyErrors=True)

  versionFile = VersionFile('')
  buildIdFile = BuildIdFile()

  print(' * %s' % versionFile.statusMessage())
  versionFile.writeFile()
  print(' * %s' % buildIdFile.statusMessage())
  buildIdFile.writeFile()

  if appSetup.hasFrontEnd():

    Typescript.compileAppJs(includeUnitTests)

    tsSrcFileNames = Typescript.findTsFiles(includeUnitTests)
    restartFileNames.extend(tsSrcFileNames)

    restartFileNames.extend(Typescript.findScssFiles())

    ## TODO Is this needed?  If not, erase functions below as well.
    #srcFileNames = updateHtml()
    #restartFileNames.extend(srcFileNames)
    #updateHtmlLinks(isDebug)

    # Supress unused variable warning.TODO delete if not compiling html anymore.
    isDebug = isDebug # pylint: disable=self-assigning-variable

  app = Flask(appName, static_folder=webRoot, static_url_path='')

  # Decrease the cache time from 12 hours to 1 second.
  app.config['SEND_FILE_MAX_AGE_DEFAULT'] = 1

  print(' * Configuring actions...')

  processor.registerWithFlask(app)

  processor.registerRoot(app, webRoot, appName + '.html')

  # This tells Flask to accept connections from anywhere.
  # Use 127.0.0.1 to allow connections from localhost only.
  host = '0.0.0.0'

  print(' * Running Flask.')

  app.run(host=host, port=port, debug=True, processes=7, threaded=False,
          extra_files=restartFileNames)

###-------------------------------------------------------------------
##def updateHtml(homeDir):
##  '''Update the generated html files.
##  '''
##
##  srcWebappRoot = homeDir + '/src/webapp'
##  buildRoot = homeDir + '/build'
##  buildHtmlRoot = buildRoot + '/html'
##
##  if not os.path.isdir(buildHtmlRoot + '/debug'):
##    os.makedirs(buildHtmlRoot + '/debug')
##  if not os.path.isdir(buildHtmlRoot + '/release'):
##    os.makedirs(buildHtmlRoot + '/release')
##
##  text = Compress.genDebugHtml(srcWebappRoot, 'ram.html',
##                               'ramIncludes.txt', 'extIncludes.txt')
##  text = text.encode('utf-8')
##  open(buildHtmlRoot + '/debug/ram.html', 'w').write(text)
##
## text = Compress.genReleaseHtml(srcWebappRoot, 'ram.html', 'ram.js', 'ext.js')
##  text = text.encode('utf-8')
##  open(buildHtmlRoot + '/release/ram.html', 'w').write(text)
##
##  text, sourceMapText, jsFileNames = Compress.genJsFile(srcWebappRoot,
##                                                        'ramIncludes.txt',
##                                                        'ram.js')
##  text = text.encode('utf-8')
##  sourceMapText = sourceMapText.encode('utf-8')
##  open(buildHtmlRoot + '/release/ram.js', 'w').write(text)
##  open(buildHtmlRoot + '/release/ram.js.map', 'w').write(sourceMapText)
##
##  text, sourceMapText, extFileNames  = Compress.genJsFile(srcWebappRoot,
##                                                          'extIncludes.txt',
##                                                          'ext.js',
##                                                          'build/webapp')
##  text = text.encode('utf-8')
##  sourceMapText = sourceMapText.encode('utf-8')
##  open(buildHtmlRoot + '/release/ext.js', 'w').write(text)
##  open(buildHtmlRoot + '/release/ext.js.map', 'w').write(sourceMapText)
##
##  text, sourceMapText, cssFileNames = Compress.genCssFile(srcWebappRoot,
##                                                          'ramIncludes.css',
##                                                          'ram.css')
##  text = text.encode('utf-8')
##  sourceMapText = sourceMapText.encode('utf-8')
##  open(buildHtmlRoot + '/release/ram.css', 'w').write(text)
##  open(buildHtmlRoot + '/release/ram.css.map', 'w').write(sourceMapText)
##
##  includeNames = ['ramIncludes.css', 'ramIncludes.txt', 'extIncludes.txt']
##  allNames = jsFileNames + cssFileNames + includeNames
##  allNames = [ os.path.join(srcWebappRoot, x) for x in allNames ]
##
##  return allNames
##
###-------------------------------------------------------------------
##def updateHtmlLinks(debug, homeDir):
##  '''Update links to the generated html files.
##
##  @param debug Boolean Whether to use the debug or release configuration.
##  '''
##
##  buildRoot = homeDir + '/build'
##  buildHtmlRoot = buildRoot + '/html'
##
##  linkFiles = [ 'ram.html',
##                'ram.js',
##                'ram.js.map',
##                'ext.js',
##                'ext.js.map',
##                'ram.css',
##                'ram.css.map' ]
##
##  # Remove existing links.
##
##  for fileName in linkFiles:
##    fileName = os.path.join(webRoot, fileName)
##    if os.path.lexists(fileName):
##      os.unlink(fileName)
##
##  # Make new links.
##
##  srcDir = buildHtmlRoot + '/release'
##
##  if debug:
##    linkFiles = [ 'ram.html' ]
##    srcDir = buildHtmlRoot + '/debug'
##
##  for fileName in linkFiles:
##    src = os.path.join(srcDir, fileName)
##    dest = os.path.join(webRoot, fileName)
##    os.symlink(src, dest)
