#-------------------------------------------------------------------
#  ResponseEncoder.py
#
#  The ResponseEncoder class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ResponseEncoder class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .BinaryResponse import BinaryResponse
from .DownloadResponse import DownloadResponse
from .desc.JsonResponseDesc import JsonResponseDesc
from ai.axe.classJson import ClassJsonEncoder
from ai.axe.web.core import AxeException
from ai.axe.util import HttpUtil
from ai.axe.util import StringUtil
from typing import Optional
from typing import Union
from werkzeug.wrappers import Response
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ResponseEncoder:
  '''Encodes objects into HTTP response strings.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new ResponseEncoder.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def encode(valueClass: Union[type, JsonResponseDesc, None],
             value: Optional[object]) -> Response:
    '''Encodes a value to an HTTP response.

    @param valueClass The class to use to encode the value.
    @param value The value to encode.

    @return A (mime-type, response string) tuple.
    '''

    if valueClass is None:
      if not value is None:
        msg = ("Invalid HTTP response type.  Expected None, but is " +
               "actually " + StringUtil.typeName(value) + ".  Value: " +
               str(value))
        raise AxeException(msg)

      mimeType = 'text/plain'
      responseStr = ''
      response = Response(responseStr, mimetype=mimeType)

    elif valueClass == str:
      if not isinstance(value, str):
        msg = ("Invalid HTTP response type.  Expected string, but is " +
               "actually " + StringUtil.typeName(value) + ".  Value: " +
               str(value))
        raise AxeException(msg)

      mimeType = 'text/plain'
      responseStr = value
      response = Response(responseStr, mimetype=mimeType)

    elif valueClass in (BinaryResponse, DownloadResponse):
      assert isinstance(valueClass, type), valueClass
      if not isinstance(value, valueClass):
        msg = ("Invalid HTTP response type.  Expected class " +
                valueClass.__name__ + ", but is " +
               "actually " + StringUtil.typeName(value) + ".  Value: " +
               str(value))
        raise AxeException(msg)

      assert isinstance(value, (BinaryResponse, DownloadResponse))
      response = value.createHttpResponse()
      HttpUtil.addNoCacheHeader(response)

    elif valueClass == Response:
      if not isinstance(value, Response):
        msg = ("Invalid HTTP response type.  Expected Response, but is " +
               "actually " + StringUtil.typeName(value) + ".  Value: " +
               str(value))
        raise AxeException(msg)

      return value

    elif isinstance(valueClass, JsonResponseDesc):
      if not isinstance(value, dict):
        msg = ("Invalid HTTP response type.  The action has a " +
               "JsonResponseDesc, so expected a dictionary object, " +
               "but is actually " + StringUtil.typeName(value) + ".  " +
               "Value: " + str(value))
        raise AxeException(msg)

      jsonResponseDesc = valueClass
      valueDict = value

      expectedAttrs = {f.name for f in jsonResponseDesc.fields}
      actualAttrs = set(valueDict.keys())
      extra = actualAttrs - expectedAttrs
      missing = expectedAttrs - actualAttrs

      if extra or missing:
        msgLines = ["Invalid HTTP response dict.  Keys are not correct."]
        if extra:
          msgLines.append("Extra keys: " + ",".join(extra))
        if missing:
          msgLines.append("Missing keys: " + ",".join(missing))
        msg = '\n'.join(msgLines)
        raise AxeException(msg)

      # Convert the value dictionary to a @ClassJsonClass-decorated object.
      valueObj = jsonResponseDesc.createObject(valueDict)

      mimeType = 'application/json'
      responseStr = ClassJsonEncoder().encode(valueObj)
      response = Response(responseStr, mimetype=mimeType)

    else:
      if not isinstance(value, valueClass):
        msg = ("Invalid HTTP response type.  Expected " + valueClass.__name__  +
               ", but is actually " + StringUtil.typeName(value) + ".  " +
               "Value: " + str(value))
        raise AxeException(msg)

      mimeType = 'application/json'
      responseStr = ClassJsonEncoder().encode(value)
      response = Response(responseStr, mimetype=mimeType)

    return response
