#-------------------------------------------------------------------
#  RequestProcessor.py
#
#  The RequestProcessor class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the RequestProcessor class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .desc.DetailDocWriter import DetailDocWriter
from .desc.WebDesc import WebDesc
from .desc.WebDescApiDocWriter import WebDescApiDocWriter
from .desc.WebDescAppDocWriter import WebDescAppDocWriter
from .ActionProcessor import ActionProcessor
from .AuthMgr import AuthMgr
from ai.axe.db.DatabaseMgr import DatabaseMgr
from ai.axe.classJson import ClassJsonSchemaWriter
from ai.axe.web.app import AppSetup
from ai.axe.web.core import AxeException
from ai.axe.web.core import Log
from flask import Flask
from typing import Type
from werkzeug.wrappers import Request
from werkzeug.wrappers import Response
import flask
import traceback
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class RequestProcessor:
  '''Handles HTTP requests.
  '''

  #-----------------------------------------------------------------
  def __init__(self, noisyErrors: bool) -> None:
    '''Creates a new RequestProcessor.

    @param noisyErrors If true, exceptions will be written to stderr.
    '''

    # Initialize the webapp.
    appSetup: AppSetup = AppSetup.get()
    appSetup.initWebFull(noisyErrors)

    # The AuthMgr subclass that will manage authentication and authorization.
    self.authMgrClass: Type[AuthMgr] = appSetup.webAuthMgrClass()

    # The DB Manager that makes sessions.
    self.databaseMgr: DatabaseMgr = appSetup.databaseMgr()

  #-----------------------------------------------------------------
  def registerWithFlask(self, app: Flask) -> None:
    '''Register this processor with the specified Flask app.
    '''

    app.add_url_rule('/app/<path:path>',
                     None,
                     self.processRequestApp,
                     methods=('GET', 'POST'))

    app.add_url_rule('/api/<path:path>',
                     None,
                     self.processRequestApi,
                     methods=('GET', 'POST'))

    app.add_url_rule('/authless/<path:path>',
                     None,
                     self.processRequestApp,
                     methods=('GET', 'POST'))

    app.add_url_rule('/app/api',
                     None,
                     self.processAppApi,
                     methods=('GET', 'POST'))

    app.add_url_rule('/app/doc',
                     None,
                     self.processAppDoc,
                     methods=('GET', 'POST'))

    app.add_url_rule('/api/doc',
                     None,
                     self.processApiDoc,
                     methods=('GET', 'POST'))

    app.add_url_rule('/app/doc/html',
                     None,
                     self.processHtmlDoc,
                     methods=('GET', 'POST'))

  #-----------------------------------------------------------------
  def registerRoot(self,
                   app: Flask,
                   htmlDirectory: str,
                   htmlFile: str) -> None:
    '''Registers this processor to serve the specified root doc.
    '''

    @app.route('/')
    def serveRoot():

      path = '/'

      # Flask stores the request as a global.
      request = flask.request

      # Create the authentication/authorization manager for this request.
      authMgr = self.authMgrClass()

      # Read the HTTP session, if there is one.
      authMgr.readHttpSession(request)

      # If the user isn't logged in, redirect to the login page.
      if authMgr.loggedInRequired(path) and not authMgr.userInfo():
        return authMgr.notAuthenticatedResponse(request)

      return flask.send_from_directory(htmlDirectory, htmlFile)

    # Suppress unused variable warning.
    serveRoot = serveRoot # pylint: disable=self-assigning-variable

  #-----------------------------------------------------------------
  def processAppDoc(self) -> Response:
    '''Process an HTTP request from Flask.
    '''

    text = WebDescAppDocWriter.formatVerbDict(WebDesc.allVerbs(False))
    return Response(text, mimetype='text/plain')

  #-----------------------------------------------------------------
  def processApiDoc(self) -> Response:
    '''Process an HTTP request from Flask.
    '''

    text = WebDescApiDocWriter.formatVerbDict(WebDesc.allVerbs(True))
    return Response(text, mimetype='text/html')

  #-----------------------------------------------------------------
  def processHtmlDoc(self) -> Response:
    '''Process an HTTP request from Flask.
    '''

    writer = DetailDocWriter()
    text = writer.formatHTMLVerbDict(WebDesc.allVerbs(False))
    return Response(text, mimetype='text/html')

  #-----------------------------------------------------------------
  def processAppApi(self) -> Response:
    '''Process an HTTP request from Flask.
    '''

    text = ClassJsonSchemaWriter().writeAllJson()
    return Response(text, mimetype='text/plain')

  #-----------------------------------------------------------------
  def processRequestApp(self, path: str) -> Response:
    '''Process an HTTP request from Flask.
    '''

    return self.processRequest(False, path)

  #-----------------------------------------------------------------
  def processRequestApi(self, path: str) -> Response:
    '''Process an HTTP request from Flask.
    '''

    return self.processRequest(True, path)

  #-----------------------------------------------------------------
  def processRequest(self,
                     isApiRequest: bool,
                     path: str) -> Response:
    '''Process an HTTP request from Flask.
    '''

    try:

      # Flask stores the request as a global.
      request: Request = flask.request

      # Create the authentication/authorization manager for this request.
      authMgr: AuthMgr = self.authMgrClass()

      # Read the HTTP session, if there is one.
      authMgr.readHttpSession(request)

      # If the user isn't logged in, redirect to the login page.
      if authMgr.loggedInRequired(path) and not authMgr.userInfo():
        if isApiRequest:
          return Response('You must log in.', 403)
        else:
          return authMgr.notAuthenticatedResponse(request)

      actionFunc = WebDesc.findAction(isApiRequest, path)
      if not actionFunc:
        msg = ('Failed to find a match for path: ' + path + '. ' +
               'Valid URLs:\n\n' + '\n'.join(WebDesc.allUrls(isApiRequest)))
        if isApiRequest:
          # For an API call, return a Bad Request 400.
          return Response(msg, 400)
        else:
          # For an app call, an exception will make a 500 and error log msg.
          raise AxeException(msg)

      actionProcessor = ActionProcessor(isApiRequest, request, authMgr,
                                         actionFunc, self.databaseMgr)
      response = actionProcessor.process()

      # Write the session to the response so it will be found on later requests.
      authMgr.writeHttpSession(response)

      return response

    # pylint: disable = W0703
    except Exception:

      msg = "Request failed for path:  " + str(path)
      exceptionStr = traceback.format_exc()
      Log.logException(msg, exceptionStr)

      return Response('Server error.', 500)
