#-------------------------------------------------------------------
#  RequestParamEncoding.py
#
#  The RequestParamEncoding class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the RequestParamEncoding class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .ParamValueException import ParamValueException
from datetime import date
from datetime import datetime
from datetime import timedelta
from ai.axe.web.core import AxeException
from ai.axe.classJson import ClassJsonDecoder
from ai.axe.classJson.jsonTypes import JsonDate
from ai.axe.classJson.jsonTypes import JsonDateTime
from ai.axe.classJson.jsonTypes import JsonDuration
from ai.axe.classJson.jsonTypes import JsonObj
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
from typing import Sized
from typing import Tuple
from typing import Union
from werkzeug.datastructures import FileStorage
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class RequestParamEncoding:
  '''How to extract a URL query parameter from a string.
  '''

  INT: 'RequestParamEncoding'
  INT_LIST: 'RequestParamEncoding'
  FLOAT: 'RequestParamEncoding'
  FLOAT_LIST: 'RequestParamEncoding'
  BOOLEAN: 'RequestParamEncoding'
  BOOLEAN_LIST: 'RequestParamEncoding'
  DATE: 'RequestParamEncoding'
  DATE_LIST: 'RequestParamEncoding'
  DATETIME: 'RequestParamEncoding'
  DATETIME_LIST: 'RequestParamEncoding'
  DURATION: 'RequestParamEncoding'
  DURATION_LIST: 'RequestParamEncoding'
  STRING: 'RequestParamEncoding'
  STRING_LIST: 'RequestParamEncoding'
  FILE: 'RequestParamEncoding'
  FILE_LIST: 'RequestParamEncoding'
  OBJECT: 'RequestParamEncoding'
  OBJECT_LIST: 'RequestParamEncoding'

  #-----------------------------------------------------------------
  # All available enums.
  allValues: List['RequestParamEncoding'] = []

  #-----------------------------------------------------------------
  # All available enums, except ClassJson objects.
  allValuesNoClassJson: List['RequestParamEncoding'] = []

  #-----------------------------------------------------------------
  def __init__(self, name: str,
               classType: Union[List[type], type],
               typescriptLabel: str) -> None:
    '''Creates a new RequestParamEncoding.
    '''

    # A string that describes what type this encodes (such as 'int').
    self.name: str = name

    # A class that describes what type this encodes (such as int).
    self.classType: Union[List[type], type] = classType

    # The typescript type equivalent to this type (such as 'number').
    self.tsLabel: str = typescriptLabel

    RequestParamEncoding.allValues.append(self)

    if name not in ('obj', 'obj_list'):
      RequestParamEncoding.allValuesNoClassJson.append(self)

  #-----------------------------------------------------------------
  def __repr__(self) -> str:

    params = ['name', 'classType', 'tsLabel']

    return StringUtil.formatRepr(self, params)

  #-----------------------------------------------------------------
  @staticmethod
  def forType(encodingType: Union[str, type, List[type]],
              objClass: Optional[type],
              paramName: str) -> Tuple['RequestParamEncoding', Optional[type]]:
    '''Returns the enum value that corresponds to the specified class type.

    @param encodingType The type of the encoding, either a string like 'int',
                        or a type object like int.
    @param objClass If encodingType is 'obj' or 'obj_list', this should be
                    the object class.
    @paramName The name of the @WebParam being processed.

    @return A (RequestParamEncoding, objClass) tuple.
    '''

    # Handle a type that's really a string name.
    if isinstance(encodingType, str):
      return RequestParamEncoding.forName(encodingType), objClass

    # Check primitives.
    for encoding in RequestParamEncoding.allValuesNoClassJson:
      if RequestParamEncoding.sameType(encodingType, encoding.classType):
        return encoding, None

    # Check objects.
    objEncoding, objClass = RequestParamEncoding.getObjClass(encodingType,
                                                             objClass,
                                                             paramName)
    if objEncoding is not None:
      return objEncoding, objClass

    # No encoding exists.  Throw an exception.

    typeStrs = []
    for encoding in RequestParamEncoding.allValuesNoClassJson:
      typeStrs.append(RequestParamEncoding.typeStr(encoding.classType))
    typeStrs.extend(['@ClassJsonClass', '[@ClassJsonClass]'])

    msg = "Error.  Invalid @WebParam '" + str(paramName) + "' encoding type: "
    msg += str(encodingType) + "  Valid types are: "
    msg += ", ".join(typeStrs)
    msg += "  User-created classes must be decorated with the @ClassJsonClass "
    msg += "decorator to be used as a @WebParam."
    raise AxeException(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def sameType(type1: Union[List[type], type],
               type2: Union[List[type], type]) -> bool:
    '''Returns true if the 2 types are the same.
    '''

    if isinstance(type1, list) or isinstance(type2, list):
      if not (isinstance(type1, list) and isinstance(type2, list)):
        return False
      else:
        RequestParamEncoding.checkTypeList(type1)
        RequestParamEncoding.checkTypeList(type2)

        return RequestParamEncoding.sameType(type1[0], type2[0])

    else:
      return type1 == type2

  #-----------------------------------------------------------------
  @staticmethod
  def getObjClass(typeClass: Union[type, List[type], str],
                  objClass: Optional[type],
                  paramName: str) -> Union[Tuple['RequestParamEncoding', type],
                                           Tuple[None, None]]:
    '''If the typeClass is an object returns the object class.
    '''

    if isinstance(typeClass, str) and typeClass in ('obj', 'obj_list'):

      if objClass is None:

        msg = "Error in @WebParam " + str(paramName) + ". The type is object, "
        msg += "but no objClass was supplied."
        raise AxeException(msg)

      elif not JsonObjRegistry.classIsRegistered(objClass):

        msg = "Error in @WebParam " + str(paramName) + ". The objClass is not "
        msg += "a ClassJson class.  The objClass is: " + str(objClass)
        raise AxeException(msg)

      elif typeClass == 'obj':
        return RequestParamEncoding.OBJECT, objClass

      elif typeClass == 'obj_list':
        return RequestParamEncoding.OBJECT_LIST, objClass

      else:
        raise AssertionError('This should be unreachable.')

    elif (isinstance(typeClass, type) and
          JsonObjRegistry.classIsRegistered(typeClass)):
      return RequestParamEncoding.OBJECT, typeClass

    elif (isinstance(typeClass, list) and
          isinstance(typeClass[0], type) and
          JsonObjRegistry.classIsRegistered(typeClass[0])):
      return RequestParamEncoding.OBJECT_LIST, typeClass[0]

    else:
      return None, None

  #-----------------------------------------------------------------
  @staticmethod
  def checkTypeList(typeList: Sized) -> None:
    '''Check that the type list has only one element.
    '''

    if len(typeList) != 1:
      msg = "Unexpected type.  List has %s elements, but expected one.  "
      msg += "Type:  %s"
      msg = msg % (len(typeList), typeList)
      raise AxeException(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def typeStr(classType: Union[List[type], type]) -> str:
    '''Returns a pretty string for a classType.

    For example int return 'int' and [int] returns '[int]'.
    '''

    if isinstance(classType, list):
      assert classType, 'List must have exactly one item.'
      return '[' + RequestParamEncoding.typeStr(classType[0]) + ']'
    else:
      return classType.__name__

  #-----------------------------------------------------------------
  @staticmethod
  def forName(encodingName: str) -> 'RequestParamEncoding':
    '''Returns the enum value that corresponds to the specified name.
    '''

    for encoding in RequestParamEncoding.allValues:
      if encoding.name == encodingName:
        return encoding

    msg = "Error.  Invalid request param encoding name: " + encodingName
    msg += "  Valid names are: "
    msg += ", ".join([x.name for x in RequestParamEncoding.allValues])
    raise AxeException(msg)

  #-----------------------------------------------------------------
  # pylint: disable = R0911, R0912
  def scalarStringToObject(self,
                           value: str,
                           objClass: Optional[type]) -> Optional[object]:
    '''Converts a parameter string to an object.

    @param value The value to decode.
    @param objClass The object class to expect if the parameter type is object.
                    Ignored if the parameter type is not object.
    '''

    if self.isList():
      msg = ("Error: called single-value stringToObject for list type " +
        self.name + ".  Value: " + value)
      raise AxeException(msg)

    try:

      if self == RequestParamEncoding.INT:
        return int(value) if value else None

      elif self == RequestParamEncoding.FLOAT:
        return float(value) if value else None

      elif self == RequestParamEncoding.BOOLEAN:
        return value == 'true' if value else None

      elif self == RequestParamEncoding.STRING:
        return value

      elif self == RequestParamEncoding.DATE:
        try:
          return JsonDate.dateStringToObject(value)
        except Exception:
          msg = ("Value '" + value + "' is not a valid date.  " +
                 "Expected format is: " + JsonDate.DATE_PATTERN)
          raise ParamValueException(msg)

      elif self == RequestParamEncoding.DATETIME:
        try:
          return JsonDateTime.datetimeStringToObject(value)
        except Exception:
          msg = ("Value '" + value + "' is not a valid datetime.  " +
                 "Expected format is: " + JsonDateTime.DATETIME_PATTERN)
          raise ParamValueException(msg)

      elif self == RequestParamEncoding.DURATION:
        if value == '':
          return None
        else:
          try:
            valueInt: int = int(value)
          except Exception:
            msg = ("Value need to be an integer number of milliseconds, " +
                   "but is actually:  " + str(value))
            raise ParamValueException(msg)
          return JsonDuration().decode(valueInt)

      elif self == RequestParamEncoding.FILE:

        # Werkzeug has already created a FileStorage object.  Just return it.
        return value

      elif self == RequestParamEncoding.OBJECT:
        valueObj = ClassJsonDecoder().decodeObject(value)

        assert objClass is not None

        if not isinstance(valueObj, objClass):
          msg = ("The object was supposed to be class '" +
                 str(objClass) + "', but was the wrong class: " +
                 str(valueObj.__class__) + "  Parameter value: " + value)
          raise ParamValueException(msg)

        return valueObj

      else:
        raise AxeException("Invalid enum: " + str(self))

    except ValueError as ex:

      msg = "Invalid value '" + value + "': " + str(ex)
      raise ParamValueException(msg)

  #-----------------------------------------------------------------
  def stringToObject(self,
                     valueArray: List[str],
                     objClass: Optional[type]) -> Optional[object]:
    '''Converts a parameter string to an object.

    @param valueArray The value to decode.
    @param objClass The object class to expect if the parameter type is object.
                    Ignored if the parameter type is not object.
  '''

    assert isinstance(valueArray, list)

    if not self.isList():
      # Non-list types use a single value from the array.  Use the last one.
      value = valueArray[-1]
      return self.scalarStringToObject(value, objClass)

    try:

      if self == RequestParamEncoding.STRING_LIST:
        return valueArray

      elif self == RequestParamEncoding.INT_LIST:
        valueIntList = [int(value) for value in valueArray]
        return valueIntList

      elif self == RequestParamEncoding.FLOAT_LIST:
        valueFloatList = [float(value) for value in valueArray]
        return valueFloatList

      elif self == RequestParamEncoding.BOOLEAN_LIST:
        valueBoolList = [(value == 'true') for value in valueArray]
        return valueBoolList

      elif self in (RequestParamEncoding.DATE_LIST,
                    RequestParamEncoding.DATETIME_LIST,
                    RequestParamEncoding.DURATION_LIST):

        valueObjList: List[object] = []
        for value in valueArray:
          valueObjList.append(self.scalarStringToObject(value, objClass))
        return valueObjList

      elif self == RequestParamEncoding.OBJECT_LIST:

        assert objClass is not None

        decoder = ClassJsonDecoder()
        valueObjList = []
        for i, valueStr in enumerate(valueArray):
          valueObj = decoder.decodeObject(valueStr)
          valueObjList.append(valueObj)

          if valueObj.__class__ != objClass:
            msg = ("The object was supposed to be class '" +
                   objClass.__name__ + "', but was the wrong class: " +
                   valueObj.__class__.__name__ + "  Parameter value " +
                   str(i) + " from array: " + str(valueArray))
            raise ParamValueException(msg)

        return valueObjList

      else:
        raise AxeException("Invalid enum: " + str(self))

    except ValueError:
      msg = "Value '" + str(valueArray) + "' is not a valid number list."
      raise ParamValueException(msg)

  #-----------------------------------------------------------------
  def extractUploadedFiles(self,
                      fileList: List[FileStorage]) -> Union[FileStorage,
                                                            List[FileStorage]]:
    '''Extracts uploaded files for this parameter.

    @param fileList A list of FileStorage objects.

    @return A FileStorage or list of FileStorage objects.
    '''

    assert self.isFile(), self

    return fileList if self.isList() else fileList[0]

  #-----------------------------------------------------------------
  def isFile(self) -> bool:
    '''Returns true if this is a file type.
    '''

    return self in (RequestParamEncoding.FILE, RequestParamEncoding.FILE_LIST)

  #-----------------------------------------------------------------
  def isList(self) -> bool:
    '''Returns true if this is a list type.
    '''

    return self in (RequestParamEncoding.STRING_LIST,
                    RequestParamEncoding.INT_LIST,
                    RequestParamEncoding.FLOAT_LIST,
                    RequestParamEncoding.BOOLEAN_LIST,
                    RequestParamEncoding.DATE_LIST,
                    RequestParamEncoding.DATETIME_LIST,
                    RequestParamEncoding.DURATION_LIST,
                    RequestParamEncoding.FILE_LIST,
                    RequestParamEncoding.OBJECT_LIST)

  #-----------------------------------------------------------------
  def label(self, objClass: Optional[type]) -> str:
    '''Returns a label suitible for display in API docs.

    @param objClass The object class to expect if the parameter type is object.
                    Ignored if the parameter type is not object.
    '''

    if not objClass:
      return self.name
    elif self.isList():
      return "list:" + objClass.__name__
    else:
      return objClass.__name__

  #-----------------------------------------------------------------
  def typescriptLabel(self,
                      objPackage: str,
                      objClass: Optional[type]) -> str:
    '''Returns a label suitible for display in typescript docs.

    @param objPackage The package name to prefix class names with.
    @param objClass The object class to expect if the parameter type is object.
                    Ignored if the parameter type is not object.
    '''

    if not objClass:

      return self.tsLabel

    else:

      classNames: List[str] = self.typescriptSubclasses(objClass)

      if objPackage:
        for i in range(len(classNames)):
          classNames[i] = objPackage + '.' + classNames[i]

      classNameStr = ' | '.join(classNames)

      if self.isList():
        classNameStr = "Array<" + classNameStr + ">"

      return classNameStr

  #-----------------------------------------------------------------
  def typescriptSubclasses(self,
                           objClass: Optional[type]) -> List[str]:
    '''Returns all the typescript classes that are substituable for this one.

    @param objClass The object class to find subclasses for.

    @return The specified class and all its subclasses.
    '''

    if objClass is None:
      return []

    jsonObj = JsonObjRegistry.getForClass(objClass)
    subclasses: List[JsonObj] = JsonObjRegistry.subclasses(jsonObj)

    classNames: List[str] = []

    assert not jsonObj.desc.isAbstract or subclasses, objClass

    if not jsonObj.desc.isAbstract:
      classNames.append(jsonObj.desc.className)

    for subclass in subclasses:
      classNames.append(subclass.desc.className)

    return classNames

#-----------------------------------------------------------------
# RequestParamEncoding enum values.

RequestParamEncoding.INT = RequestParamEncoding('int', int, 'number')
RequestParamEncoding.INT_LIST = RequestParamEncoding('int_list', [int],
                                                     'Array<number>')

RequestParamEncoding.FLOAT = RequestParamEncoding('float', float, 'number')
RequestParamEncoding.FLOAT_LIST = RequestParamEncoding('float_list', [float],
                                                       'Array<number>')

RequestParamEncoding.BOOLEAN = RequestParamEncoding('boolean', bool, 'boolean')
RequestParamEncoding.BOOLEAN_LIST = RequestParamEncoding('boolean_list', [bool],
                                                         'Array<boolean>')

RequestParamEncoding.DATE = RequestParamEncoding('date', date, 'Day')
RequestParamEncoding.DATE_LIST = RequestParamEncoding('date_list', [date],
                                                      'Array<Day>')

RequestParamEncoding.DATETIME = RequestParamEncoding('datetime',
                                                     datetime,
                                                     'Date')
RequestParamEncoding.DATETIME_LIST = RequestParamEncoding('datetime_list',
                                                          [datetime],
                                                          'Array<Date>')

RequestParamEncoding.DURATION = RequestParamEncoding('duration',
                                                     timedelta,
                                                     'Duration')
RequestParamEncoding.DURATION_LIST = RequestParamEncoding('duration_list',
                                                          [timedelta],
                                                          'Array<Duration>')

RequestParamEncoding.STRING = RequestParamEncoding('string', str, 'string')
RequestParamEncoding.STRING_LIST = RequestParamEncoding('string_list', [str],
                                                        'Array<string>')
RequestParamEncoding.FILE = RequestParamEncoding('file', FileStorage, 'File')
RequestParamEncoding.FILE_LIST = RequestParamEncoding('file_list',
                                                      [FileStorage],
                                                      'FileList')

RequestParamEncoding.OBJECT = RequestParamEncoding('obj', object, '')
RequestParamEncoding.OBJECT_LIST = RequestParamEncoding('obj_list', [object],
                                                        '')
