#-------------------------------------------------------------------
#  DownloadResponse.py
#
#  The DownloadResponse class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the DownloadResponse class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from typing import Union
from werkzeug.wrappers import Response
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class DownloadResponse:
  '''An HTTP response that is a download to the browser.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               data: Union[str, bytes],
               fileName: str,
               mimeType: str,
               isText: bool = False) -> None:
    '''Creates a new DownloadResponse.
    '''

    # The data to be downloaded.
    self.data: Union[str, bytes] = data

    # The mime type of the downloaded file.
    self.mimeType: str = mimeType

    # The file name the browser will use to save the file.
    self.fileName: str = StringUtil.sanitizeFileName(fileName)

    # Whether the browser should always show a 'save as' dialog.
    self.forceSaveAs: bool = True

    assert (isText and isinstance(data, str) or
            (not isText) and isinstance(data, bytes)), (isText, type(data))

    # If the download is text, it should be UTF-8 encoded.
    if isinstance(self.data, str):
      self.data = self.data.encode('utf-8')

  #-----------------------------------------------------------------
  def createHttpResponse(self) -> Response:
    '''Writes the download to an HTTP response.
    '''

    response = Response(self.data, mimetype=self.mimeType)

    # Set the response headers.

    numBytes = len(self.data)
    response.headers.add('Content-Length', numBytes)

    # If the Content-Disposition is 'inline' the browser will display
    # the response in the browser if possible; otherwise the browser
    # will display the 'Save As' dialog.
    #
    # If the Conent-Disposition is 'attachment', the browser will always
    # display the 'Save As' dialog.

    disposition = "inline"
    if self.forceSaveAs:
      disposition = "attachment"

    disposition += "; filename=\"" + self.fileName + "\""

    response.headers.add("Content-Disposition", disposition)

    return response
