#-------------------------------------------------------------------
#  TestStringUtil.py
#
#  The TestStringUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the StringUtil class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.util import StringUtil
from importlib import reload
from typing import Dict
from typing import List
from typing import Union
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestStringUtil(AxeSimpleTestCase):
  '''Unit test for the StringUtil class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testImport(self) -> None:
    '''Test importing StringUtil.
    '''

    # Because the build system depends on StringUtil,
    # StringUtil will already have been imported before its unit test
    # begins to run, and the coverage tool marks the 'def' lines
    # as not covered.
    #
    # This line will let the coverage tool see the module has been imported.

    reload(StringUtil)

  #-----------------------------------------------------------------
  def testFormatDict(self) -> None:
    '''Test formatting a dictionary.
    '''

    theDict: Dict[str, List[str]] = OrderedDict()
    theDict['a'] = ['1']
    theDict['b'] = ['1', '2', '3']
    theDict['password'] = ['something']

    actual = StringUtil.formatMultiDict(theDict, 'pa.*.*word')

    expected = 'a=1, b=1,2,3, password=[value_not_shown]'

    self.assertEqual(expected, actual, 'dict')

  #-----------------------------------------------------------------
  def testFormatRepr(self) -> None:
    '''Test the formatRepr() method.
    '''

    # Test 3 cases for extraMembers:
    #
    # "b" is not present on the object, but is in showMembers.
    # "c" is present on the object, and also in showMembers and extraMembers.
    # "d" is present on the object, and also in extraMembers.

    class Foo:
      def __init__(self, a, c):
        #pylint: disable=C0103
        self.a = a
        self.c = c

    f = Foo(1, -1)

    expected = "Foo(a=1, b=2, c=3, d=4)"
    actual = StringUtil.formatRepr(f,
                                   ['a', 'b', 'c'],
                                   [("b", 2), ("c", 3), ("d", 4)])

    self.assertEqual(expected, actual, 'repr')

  #-----------------------------------------------------------------
  def testTypeName(self) -> None:
    '''Test getting an object type name
    '''

    class Foo:
      def __init__(self) -> None:
        pass

    self.assertEqual('int', StringUtil.typeName(3), 'int')
    self.assertEqual('Foo', StringUtil.typeName(Foo()), 'Foo')

  #-----------------------------------------------------------------
  def testSanitizeFileName(self) -> None:
    '''Test sanitizing a file name.
    '''

    badName = r'a b c x/y\,z.txt'

    expected = 'a_b_c_xyz.txt'

    actual = StringUtil.sanitizeFileName(badName)

    self.assertEqual(expected, actual, 'sanitized')

  #-----------------------------------------------------------------
  def testFormatPostres2Array(self) -> None:
    '''Test formatting a Postgres array string.
    '''

    raw: List[List[Union[float, None]]] = [[]]
    expected = "ARRAY[[]]"
    actual = StringUtil.formatPostgresArray2(raw)

    self.assertEqual(expected, actual, 'empty')

    raw = [[1, 2, None, 3]]
    expected = 'ARRAY[[1,2,NULL,3]]'
    actual = StringUtil.formatPostgresArray2(raw)

    self.assertEqual(expected, actual, 'single')

    raw = [[1, 2, 3], [4, 5, 6]]
    expected = 'ARRAY[[1,2,3],[4,5,6]]'
    actual = StringUtil.formatPostgresArray2(raw)

    self.assertEqual(expected, actual, 'multi')

    raw = [[1.0, 2.0, 3.0], [4.0, 5.0, 6.0]]
    expected = 'ARRAY[[1.0,2.0,3.0],[4.0,5.0,6.0]]'
    actual = StringUtil.formatPostgresArray2(raw)

    self.assertEqual(expected, actual, 'multi float')

  #-----------------------------------------------------------------
  def testParsePostres2Array(self) -> None:
    '''Test converting a Postgres array string.
    '''

    raw = '{{}}'
    expected: List[List[Union[float, None]]] = [[]]
    actual = StringUtil.parsePostgresArray2(raw, float)

    self.assertEqual(expected, actual, 'empty')

    raw = '{{1,2,NULL,3}}'
    expected = [[1, 2, None, 3]]
    actual = StringUtil.parsePostgresArray2(raw, int)

    self.assertEqual(expected, actual, 'single')

    raw = '{{1,2,3},{4,5,6}}'
    expected = [[1, 2, 3], [4, 5, 6]]
    actual = StringUtil.parsePostgresArray2(raw, int)

    self.assertEqual(expected, actual, 'multi')

    raw = '{{1,2,3},{4,5,6}}'
    expected = [[1.0, 2.0, 3.0], [4.0, 5.0, 6.0]]
    actual = StringUtil.parsePostgresArray2(raw, int)

    self.assertEqual(expected, actual, 'multi float')

  #-----------------------------------------------------------------
  def testFormatBigNumber(self) -> None:
    '''Test converting large numbers to comma separated strings
    '''

    expected = "100"
    actual = StringUtil.formatBigNumber(100)

    self.assertEqual(expected, actual)

    expected = "1,000"
    actual = StringUtil.formatBigNumber(1000)

    self.assertEqual(expected, actual)

    expected = "1,000,000"
    actual = StringUtil.formatBigNumber(1000000)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testCapitalize(self) -> None:
    '''Test capitalizing a word.
    '''

    expected = "Foo"
    actual = StringUtil.capitalize('foo')

    self.assertEqual(expected, actual)

    expected = ""
    actual = StringUtil.capitalize("")

    self.assertEqual(expected, actual)

    expected = "X"
    actual = StringUtil.capitalize("x")

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testLower(self) -> None:
    '''Test lower-casing a word.
    '''

    expected = "foo"
    actual = StringUtil.initialLower('Foo')

    self.assertEqual(expected, actual)

    expected = ""
    actual = StringUtil.initialLower("")

    self.assertEqual(expected, actual)

    expected = "x"
    actual = StringUtil.initialLower("X")

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testCamelCaseToUnderscore(self) -> None:
    '''Test converting camel case to underscores.
    '''

    data = (
      ('', ''),
      ('a', 'a'),
      ('A', 'a'),
      ('apple', 'apple'),
      ('Apple', 'apple'),
      ('appleAndBanana', 'apple_and_banana'),
      ('AppleAndBanana', 'apple_and_banana'),
      )

    for camelCase, expected in data:
      actual = StringUtil.camelCaseToUnderscores(camelCase)
      self.assertEqual(expected, actual, msg=camelCase)

  #-----------------------------------------------------------------
  def testUnderscoresToCamelCase(self) -> None:
    '''Test converting underscores to camel case.
    '''

    data = (
      ('', False, ''),
      ('', True, ''),
      ('a', False, 'a'),
      ('a', True, 'A'),
      ('apple', False, 'apple'),
      ('apple', True, 'Apple'),
      ('apple_and_banana', False, 'appleAndBanana'),
      ('apple_____and_banana', True, 'AppleAndBanana'),
      )

    for underscores, initialCapital, expected in data:
      actual = StringUtil.underscoresToCamelCase(underscores, initialCapital)
      self.assertEqual(expected, actual,
                       msg=('%s %s' % (underscores, initialCapital)))

  #-----------------------------------------------------------------
  def testIsUuid(self) -> None:
    '''Test detecting a UUID.
    '''

    text = ""
    label = "Empty string."
    self.assertEqual(False, StringUtil.isUuid(text), label)

    text = '0bb1637c-44d7-4341-b470-bd8b5a5a3e'
    label = "Too short."
    self.assertEqual(False, StringUtil.isUuid(text), label)

    text = '0bb1637c-44d7-4341-b470-bd8b5a5a3ed411111'
    label = "Too long."
    self.assertEqual(False, StringUtil.isUuid(text), label)

    text = ' 0bb1637c-44d7-4341-b470-bd8b5a5a3ed4'
    label = "Leading whitespace."
    self.assertEqual(False, StringUtil.isUuid(text), label)

    text = '0bb1637c-44d7-4341-b470-bd8b5a5a3ed4 '
    label = "Trailing whitespace."
    self.assertEqual(False, StringUtil.isUuid(text), label)

    text = '0bb1637c-44d7-4341-b470-bd8b5a5a3ed4'
    label = "Valid UUID."
    self.assertEqual(True, StringUtil.isUuid(text), label)

    text = '0-bb16-37c44d74341-b470bd8b5a5a3ed4'
    label = "Dashes in strange places."
    self.assertEqual(True, StringUtil.isUuid(text), label)

    text = '0BB1637C-44d7-4341-B470-BD8b5A5A3ED4'
    label = "Upper-case UUID."
    self.assertEqual(True, StringUtil.isUuid(text), label)

    text = '0bb1637c-44d7-5341-b470-bd8b5a5a3ed4'
    label = "Wrong version (first number of 3rd text group)."
    self.assertEqual(False, StringUtil.isUuid(text), label)

    text = '0bb1637c-44d7-4341-c470-bd8b5a5a3ed4'
    label = "Fourth group must start with 8, 9, or b."
    self.assertEqual(False, StringUtil.isUuid(text), label)

  #-----------------------------------------------------------------
  def testIsInteger(self) -> None:
    '''Test detecting an int.
    '''

    text = ""
    label = "Empty string."
    self.assertEqual(False, StringUtil.isInteger(text), label)

    text = "33"
    label = "Number."
    self.assertEqual(True, StringUtil.isInteger(text), label)

    text = "3.3"
    label = "Float."
    self.assertEqual(False, StringUtil.isInteger(text), label)

    # The int() function allows leading and trailing whitespace,
    # so isInteger() does too.  Not sure it should...

    text = " 33"
    label = "Leading whitespace."
    self.assertEqual(True, StringUtil.isInteger(text), label)

    text = "33 "
    label = "Trailing whitespace."
    self.assertEqual(True, StringUtil.isInteger(text), label)

  #-----------------------------------------------------------------
  def testFormatFloat(self) -> None:
    '''Test formatting a float.
    '''

    self.assertEqual('0', StringUtil.formatFloat(0.0), 0.0)
    self.assertEqual('0.05', StringUtil.formatFloat(0.05), 0.05)
    self.assertEqual('0.05', StringUtil.formatFloat(0.05123), 0.05123)
    self.assertEqual('1', StringUtil.formatFloat(1.0), 1.0)

  #-----------------------------------------------------------------
  def testFormatPercent(self) -> None:
    '''Test formatting a percent.
    '''

    self.assertEqual('0 %', StringUtil.formatPercent(0.0), 0.0)
    self.assertEqual('5 %', StringUtil.formatPercent(0.05), 0.05)
    self.assertEqual('5 %', StringUtil.formatPercent(0.05123), 0.05123)
    self.assertEqual('100 %', StringUtil.formatPercent(1.0), 1.0)
