#-------------------------------------------------------------------
#  TestScale.py
#
#  The TestScale module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the Scale class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import datetime
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.util import Scale
from importlib import reload
from typing import Tuple
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestScale(AxeSimpleTestCase):
  '''Unit test for the Scale class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testImport(self) -> None:
    '''Test importing StringUtil.
    '''

    # Because the build system depends on util,
    # Scale will already have been imported before its unit test
    # begins to run, and the coverage tool marks the 'def' lines
    # as not covered.
    #
    # This line will let the coverage tool see the module has been imported.

    reload(sys.modules[Scale.__module__])

  #-----------------------------------------------------------------
  def testErrors(self) -> None:
    '''Test error messages.
    '''

    try:
      Scale.createFloat(10, 10, 100, 200)
      self.fail("No exception for zero-width domain.")
    except ValueError:
      pass

    try:
      Scale.createFloat(10, 100, 100, 100)
      self.fail("No exception for zero-width range.")
    except ValueError:
      pass

  #-----------------------------------------------------------------
  def testScale(self) -> None:
    '''Test scaling numbers.
    '''

    # A scale with a multiplier but no offset.
    scale = Scale.createFloat(10, 20, 100, 200)

    values: Tuple[Tuple[int, int], ...] = (
      (10, 100),
      (20, 200),
      (0, 0),
      (500, 5000),
      )

    self.assertValues(scale, values, "Multiplier only")

    # A scale with an offset but no multiplier.
    scale = Scale.createFloat(150, 250, 100, 200)

    values = (
      (150, 100),
      (250, 200),
      (10, -40),
      (20, -30),
      (0, -50),
      )

    self.assertValues(scale, values, "Offset only")

    # A scale with an offset and multiplier.
    scale = Scale.createFloat(-25, 25, 1001, 2001)

    values = (
      (-25, 1001),
      (25, 2001),
      (0, 1501),
      (250, 6501),
      )

    self.assertValues(scale, values, "Offset and multipler")

    # Get coverage up by checking str().
    str(scale)

  #-----------------------------------------------------------------
  def testScaleDatetime(self) -> None:
    '''Test scaling datetimes.
    '''

    d030 = datetime(2014, 3, 1, 0, 30)
    d100 = datetime(2014, 3, 1, 1)
    d130 = datetime(2014, 3, 1, 1, 30)
    d200 = datetime(2014, 3, 1, 2)
    d230 = datetime(2014, 3, 1, 2, 30)

    # A scale with a multiplier but no offset.
    scale = Scale.createDatetime(d100, d200, 100, 200)

    values = (
      (d030, 50),
      (d100, 100),
      (d130, 150),
      (d200, 200),
      (d230, 250),
      )

    self.assertValues(scale, values, "Datetimes")

  #-----------------------------------------------------------------
  def assertValues(self, scale, domainRangeList, msg) -> None:

    for domain, rangeValue in domainRangeList:

      self.assertEqual(rangeValue, scale.scale(domain),
                       msg + ' scale domain=' + str(domain))
      self.assertEqual(domain, scale.invert(rangeValue),
                       msg + ' invert range=' + str(rangeValue))
