#-------------------------------------------------------------------
#  TestFileUtil.py
#
#  The TestFileUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the FileUtil class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import os
import shutil
from ai.axe.util import FileUtil
from ai.axe.build.unittest import AxeSimpleTestCase
from io import BytesIO
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestFileUtil(AxeSimpleTestCase):
  '''Unit test for the FileUtil class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.

    if os.path.exists(self.scratchDir()):
      shutil.rmtree(self.scratchDir())

    os.makedirs(self.scratchDir())

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.

    shutil.rmtree(self.scratchDir())

  #-----------------------------------------------------------------
  def testReadLinesFromEnd(self) -> None:
    '''Test reading lines from the end of a file.
    '''

    text = '''Hello,
It's nice to see you today.

Sincerely,

Charlie Brown'''

    lines: List[str]

    # pylint: disable=unnecessary-comprehension
    lines = [x for x in FileUtil.readLinesFromEnd(BytesIO(text.encode()))]
    # pylint: enable=unnecessary-comprehension

    expected = [
      'Charlie Brown',
      '',
      'Sincerely,',
      '',
      "It's nice to see you today.",
      'Hello,',
      ]

    self.assertEqual(expected, lines)

  #-----------------------------------------------------------------
  def testRecusiveGlob(self) -> None:
    '''Test the recursiveGlob() function.
    '''

    scratchDir = self.scratchDir()

    os.makedirs(scratchDir + '/a')
    os.makedirs(scratchDir + '/b')
    os.makedirs(scratchDir + '/c')

    open(scratchDir + '/a/a.yes', 'w').write('foo')
    open(scratchDir + '/a/a.no', 'w').write('foo')
    open(scratchDir + '/b/b.yes', 'w').write('foo')
    open(scratchDir + '/b/b.no', 'w').write('foo')
    open(scratchDir + '/c/c.yes', 'w').write('foo')
    open(scratchDir + '/c/c.no', 'w').write('foo')

    actual = FileUtil.recursiveGlob(scratchDir, '*.yes')

    expected = [
      scratchDir + '/a/a.yes',
      scratchDir + '/b/b.yes',
      scratchDir + '/c/c.yes',
      ]

    self.assertEqual(expected, actual, "recursiveGlob")

  #-----------------------------------------------------------------
  def testGlobExErrors(self) -> None:
    '''Tests errors in the globEx() function.
    '''

    scratchDir = self.scratchDir()

    try:
      FileUtil.globEx(scratchDir + '/1/**/a/**/*.yes')
      self.fail("No exception for multiple ** in a pattern.")

    except ValueError:
      pass


    try:
      FileUtil.globEx(scratchDir + '/!1/!a/*/*.yes')
      self.fail("No exception for multiple ! in a pattern.")

    except ValueError:
      pass

  #-----------------------------------------------------------------
  def testGlobEx(self) -> None:
    '''Tests the globEx() function.
    '''

    scratchDir = self.scratchDir()

    os.makedirs(scratchDir + '/1/a')
    os.makedirs(scratchDir + '/1/b')
    os.makedirs(scratchDir + '/2/b')
    os.makedirs(scratchDir + '/2/c')

    open(scratchDir + '/1/a/a.yes', 'w').write('foo')
    open(scratchDir + '/1/a/a.no', 'w').write('foo')
    open(scratchDir + '/1/b/b.yes', 'w').write('foo')
    open(scratchDir + '/2/b/b.no', 'w').write('foo')
    open(scratchDir + '/2/c/c.yes', 'w').write('foo')
    open(scratchDir + '/2/c/c.no', 'w').write('foo')

    # Test zero levels deep.

    actual = FileUtil.globEx(scratchDir + '/1/a/**/*.yes')

    expected = [
      scratchDir + '/1/a/a.yes',
      ]

    self.assertEqual(expected, actual, "zero levels deep")

    # Test multiple levels deep.

    actual = FileUtil.globEx(scratchDir + '/**/*.yes')

    expected = [
      scratchDir + '/1/a/a.yes',
      scratchDir + '/1/b/b.yes',
      scratchDir + '/2/c/c.yes',
      ]

    self.assertEqual(expected, actual, "deep")

    # Multiple levels deep with negation.

    actual = FileUtil.globEx(scratchDir + '/**/!b/*.yes')

    expected = [
      scratchDir + '/1/a/a.yes',
      scratchDir + '/2/c/c.yes',
      ]

    self.assertEqual(expected, actual, "negation")

    # Negation without a **.

    actual = FileUtil.globEx(scratchDir + '/1/!b')

    expected = [
      scratchDir + '/1/a',
      ]

    self.assertEqual(expected, actual, "negation only, no **")

    # No negation or **.

    actual = FileUtil.globEx(scratchDir + '/1/b')

    expected = [
      scratchDir + '/1/b',
      ]

    self.assertEqual(expected, actual, "no negation or **")

  #-----------------------------------------------------------------
  def scratchDir(self) -> str:
    '''The name of the scratch directory that should be used for testing.
    '''

    rootDir = os.environ['AXE_HOME']

    return rootDir + "/build/fileUtilScratch"
