#-------------------------------------------------------------------
#  StringUtil.py
#
#  The StringUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''String utilities.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
from typing import Type
from typing import TypeVar
from typing import Union
from uuid import UUID
import re
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def formatMultiDict(theDict: Dict[str, List[str]], hideKeyRe: str) -> str:
  '''Formats the dictionary into a string for output.

   @param hideKeyRe The name of a key whose value should be hidden.
                    If null, nothing will be hidden.

   @return The formatted map as a string.
   '''

  hideRe = None
  if hideKeyRe:
    hideRe = re.compile(hideKeyRe)

  items = []
  for key, values in list(theDict.items()):

    isHidden = hideRe and hideRe.match(key)
    if isHidden:
      itemStr = key + "=[value_not_shown]"
    else:
      itemStr = key + "=" + ','.join(values)
    items.append(itemStr)

  return ', '.join(items)

#-------------------------------------------------------------------
def formatFloat(x: float) -> str:
  '''Formats the passed-in float as a string.

  For example, the following floats:

    0.0
    0.05
    0.05123
    1.0

  are formatted like so:

    0
    0.05
    0.05
    1
  '''

  # Trim to precision of 2.
  text = '%.2f' % x

  # Remove trailing zeros.
  text = text.rstrip('0')

  # Remove a trailing decimal point.
  text = text.rstrip('.')

  return text

#-------------------------------------------------------------------
def formatPercent(x: float) -> str:
  '''Formats the passed-in float as a whole percent.

  For example, the following floats:

    0.0
    0.05
    0.05123
    1.0

  are formatted like so:

    0 %
    5 %
    5 %
    100 %
  '''

  # Convert from 0-1 float to a 0-100 decimal.
  text = '%d' % (x * 100)

  # Add percent sign.
  text = text + ' %'

  return text

#-------------------------------------------------------------------
def formatRepr(obj: object,
               showMembers: List[str],
               extraMembers: Optional[List[Tuple[str, object]]] = None) -> str:
  '''Formats an object in a repr() style.

  @param obj The object to format.
  @param showMembers A list of names of data members to include in the string.
  @param extraMembers A list of (key, value) tuples to add to the string.
  '''

  if extraMembers is None:
    extraMembersDict: Dict[str, object] = {}
  else:
    # Make a copy so we can remove items.
    extraMembersDict = OrderedDict(extraMembers)

  memberStrs: List[str] = []

  for name in showMembers:

    value: str = "(missing)"
    if name in extraMembersDict:
      value = repr(extraMembersDict.pop(name))
    elif hasattr(obj, name):
      value = repr(getattr(obj, name))

    memberStrs.append(name + '=' + value)

  for name, valueObj in extraMembersDict.items():
    memberStrs.append(name + '=' + str(valueObj))

  membersStr: str = ', '.join(memberStrs)

  return classNameNoPackage(obj) + '(' + membersStr + ')'

#-------------------------------------------------------------------
def classNameNoPackage(obj: object) -> str:
  '''Returns the name of a class with no package.

  @param obj The object whose class name will be returned.
  '''

  return obj.__class__.__name__.split('.')[-1]

#-------------------------------------------------------------------
def typeName(value: object) -> str:
  '''Returns the type of the specified value.
  '''

  return type(value).__name__

#-------------------------------------------------------------------
def camelCaseToUnderscores(text: str) -> str:
  '''Returns a copy of the text with camel case converted to underscores.
  '''

  text = camelCaseRe1.sub(r'\1_\2', text)
  text = camelCaseRe2.sub(r'\1_\2', text)
  text = text.lower()
  return text

#-------------------------------------------------------------------
# Camel-case regexes.
camelCaseRe1 = re.compile('(.)([A-Z][a-z]+)')
camelCaseRe2 = re.compile('([a-z0-9])([A-Z])')

#-------------------------------------------------------------------
def underscoresToCamelCase(text: str, initialCapital: bool = False) -> str:
  '''Returns a copy of the text with camel case converted to underscores.

  @param text The text to convert to camel case.
  @param initialCapital Whether the first letter should be capital or
                        lower-case.
  '''

  # Collapse multiple underscores into one.
  text = manyUnderscoresRe.sub('_', text)

  words = text.split('_')

  newWords = []
  if words and not initialCapital:
    newWords.append(words.pop(0))

  for word in words:
    newWords.append(capitalize(word))

  text = ''.join(newWords)

  return text

#-------------------------------------------------------------------
# Regex to match runs of multiple underscores.
manyUnderscoresRe = re.compile('_{2,}')

#-------------------------------------------------------------------
def sanitizeFileName(fileName: str) -> str:
  '''Removes dangerous characters from a file name.

  @param fileName A file name to be sanitized.

  @return a sanitized file name.
  '''

  # Replace all white space with underbars.
  fileName = re.sub("\\s", "_", fileName)

  # Remove everything that's not a letter, number, underbar, or period.
  fileName = re.sub("[^\\w.]", "", fileName)

  return fileName




#-------------------------------------------------------------------
def formatBool(text: Union[bool, str]) -> str:
  '''Parses a bool or bool string into a bool consistent with JSON.

  @param text the bool or string to be formatted

  @return "true" or "false", depending on text
  '''

  if text is True or text == "True" or text == "true" or text == "t":
    return "true"
  else:
    return "false"

#-------------------------------------------------------------------
def formatPostgresArray2(array2d: List[List[Union[float, None]]]) -> str:
  '''Formats a 2-D list into a string.

  @param array2d The array to format.

  @return The equivalent string.
  '''

  ret = "ARRAY["

  for i, anArray in enumerate(array2d):
    if i > 0:
      ret += ","
    ret += "["
    for j, item in enumerate(anArray):
      if j > 0:
        ret += ","
      if item is None:
        ret += "NULL"
      else:
        ret += str(item)
    ret += "]"

  ret += "]"

  return ret

#-------------------------------------------------------------------
# pylint: disable=C0103
T = TypeVar('T', int, float)
# pylint: enable=C0103

#-------------------------------------------------------------------
def parsePostgresArray2(arrayStr: str,
                        numberType: Type[T]) -> List[List[Union[T, None]]]:
  '''Parses a 2-D Postgres array string into nested lists.

  @param arrayStr The Postgres array string.
  @param numberType The type of number expected (int or float).
  '''

  assert arrayStr.startswith('{{')
  assert arrayStr.endswith('}}')

  arrayStr = arrayStr[2:-2]

  if not arrayStr:
    return [[]]

  arrayStrs = arrayStr.split("},{")

  ret = []

  for anArrayStr in arrayStrs:
    valStrs = anArrayStr.split(",")
    vals: List[Union[T, None]] = []
    for valStr in valStrs:
      if valStr == 'NULL':
        vals.append(None)
      else:
        vals.append(numberType(valStr))
    ret.append(vals)

  return ret

#-----------------------------------------------------------------
def formatBigNumber(number: int) -> str:
  '''Formats a BIG number into a string with commas

  @param number the number to convert

  @return the number as a string with commas
  '''

  numberString = str(number)
  numberString = numberString[::-1]
  outputString = ""

  for i, j in enumerate(numberString):
    if i % 3 == 0 and i != 0:
      outputString += ","
    outputString += j

  return outputString[::-1]

#-----------------------------------------------------------------
def capitalize(text: str) -> str:
  '''Returns the text with the initial letter capitalized.
  '''

  if not text:
    return text

  return text[0].upper() + text[1:]

#-----------------------------------------------------------------
def initialLower(text: str) -> str:
  '''Returns the text with the initial letter lower-case.
  '''

  if not text:
    return text

  return text[0].lower() + text[1:]

#-----------------------------------------------------------------
def isUuid(text: str) -> bool:
  '''Returns true if the specified text is a valid UUID.

  @param text A string to check whether it is a UUID.

  @return True or False.
  '''

  # Remove dashes, which are optional.
  text = text.replace("-", "")

  # Ignore case.
  text = text.lower()

  try:
    uuid = UUID(text, version=4)
  except ValueError:
    return False

  # The UUID constructor sometimes auto-converts to a valid UUID.
  # Check to see if that's what happened here.
  return uuid.hex == text

#-----------------------------------------------------------------
def isInteger(text: str) -> bool:
  '''Returns true if the specified text is a valid int.

  @param text A string to check whether it is an int.

  @return True or False.
  '''

  try:
    int(text)
    return True
  except ValueError:
    return False
