#-------------------------------------------------------------------
#  Scale.py
#
#  The Scale class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the Scale class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import datetime
from datetime import timedelta
from typing import Generic
from typing import TypeVar
from typing_extensions import Protocol
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------

# Suppress unused variable warning.
unused = timedelta

#===================================================================
# Protocols that Scale Domain/Range types must implement.

ItemType = TypeVar('ItemType')
ItemDiffType = TypeVar('ItemDiffType')

# pylint: disable=W0104

class ItemProtocol(Protocol[ItemType,
                            ItemDiffType]):
  '''The operations a type must support to be a DomainType or RangeType.
  '''

  def __sub__(self: ItemType, other: ItemType) -> ItemDiffType:
    ...

  def __add__(self: ItemType, other: ItemDiffType) -> ItemType:
    ...

class ItemDiffProtocol(Protocol[ItemDiffType]):
  '''The operations a type must support to be a DomainDiffType or RangeDiffType.
  '''

  def __truediv__(self: ItemDiffType, other: ItemDiffType) -> float:
    ...

  def __mul__(self: ItemDiffType, other: float) -> ItemDiffType:
    ...

# pylint: enable=W0104

#===================================================================
DomainType = TypeVar("DomainType", bound=ItemProtocol)
DomainDiffType = TypeVar("DomainDiffType", bound=ItemDiffProtocol)
RangeType = TypeVar("RangeType", bound=ItemProtocol)
RangeDiffType = TypeVar("RangeDiffType", bound=ItemDiffProtocol)

#===================================================================
class Scale(Generic[DomainType, DomainDiffType, RangeType, RangeDiffType]):
  '''Translates values in a domain to values in a range.

  Modeled on the D3 scale class.
  '''

  # pylint: disable=W0613,W0104
  #-----------------------------------------------------------------
  @staticmethod
  def createDatetime(domainLow: datetime,
                domainHigh: datetime,
                rangeLow: float,
                rangeHigh: float) -> 'Scale[datetime, timedelta, float, float]':
    '''Returns a newly created Scale that maps from datetime to float.
    '''

    return Scale(domainLow, domainHigh, rangeLow, rangeHigh)

  #-----------------------------------------------------------------
  @staticmethod
  def createInt(domainLow: int,
                domainHigh: int,
                rangeLow: float,
                rangeHigh: float) -> 'Scale[int, int, float, float]':
    '''Returns a newly created Scale that maps from int to float.
    '''

    return Scale(domainLow, domainHigh, rangeLow, rangeHigh)

  #-----------------------------------------------------------------
  @staticmethod
  def createFloat(domainLow: float,
                  domainHigh: float,
                  rangeLow: float,
                  rangeHigh: float) -> 'Scale[float, float, float, float]':
    '''Returns a newly created Scale that maps from float to float.
    '''

    return Scale(domainLow, domainHigh, rangeLow, rangeHigh)

  # pylint: enable=W0613,W0104

  #-----------------------------------------------------------------
  def __init__(self,
               domainLow: DomainType,
               domainHigh: DomainType,
               rangeLow: RangeType,
               rangeHigh: RangeType) -> None:
    '''Creates a new Scale.
    '''

    self.domainLow = domainLow
    self.domainHigh = domainHigh
    self.rangeLow = rangeLow
    self.rangeHigh = rangeHigh

    domainWidth = self.domainWidth()
    rangeWidth = self.rangeWidth()

    if domainWidth == 0:
      msg = "Domain low and high can't be the same.  low=%s high=%s"
      msg = msg % (domainLow, domainHigh)
      raise ValueError(msg)

    if rangeWidth == 0:
      msg = "Range low and high can't be the same.  low=%s high=%s"
      msg = msg % (rangeLow, rangeHigh)
      raise ValueError(msg)

  #-----------------------------------------------------------------
  def domainWidth(self) -> DomainDiffType:
    '''Returns the width of the domain.
    '''

    return self.domainHigh - self.domainLow

  #-----------------------------------------------------------------
  def rangeWidth(self) -> RangeDiffType:
    '''Returns the width of the range.
    '''

    return self.rangeHigh - self.rangeLow

  #-----------------------------------------------------------------
  def scale(self, domainValue: DomainType) -> RangeType:
    '''Converts a domain value to a range value.

    @param domainValue A value from the scale domain.

    @return A number in the scale range.
    '''

    domainDelta: DomainDiffType = domainValue - self.domainLow
    domainWidth: DomainDiffType = self.domainHigh - self.domainLow
    deltaPercent: float = domainDelta / domainWidth

    rangeWidth: RangeDiffType = self.rangeHigh - self.rangeLow
    rangeDelta: RangeDiffType = rangeWidth * deltaPercent
    rangeValue: RangeType = self.rangeLow + rangeDelta

    return rangeValue

  #-----------------------------------------------------------------
  def invert(self, rangeValue: RangeType) -> DomainType:
    '''Converts a range value to a domain value.

    @param rangeValue A value from the scale range.

    @return A number in the scale domain.
    '''

    rangeDelta: RangeDiffType = rangeValue - self.rangeLow
    rangeWidth: RangeDiffType = self.rangeHigh - self.rangeLow
    deltaPercent: float = rangeDelta / rangeWidth

    domainWidth: DomainDiffType = self.domainHigh - self.domainLow
    domainDelta: DomainDiffType = domainWidth * deltaPercent
    domainValue: DomainType = self.domainLow + domainDelta

    return domainValue

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['domainLow', 'domainHigh', 'rangeLow', 'rangeHigh']

    return StringUtil.formatRepr(self, attrs)
