#-------------------------------------------------------------------
#  FileUtil.py
#
#  The FileUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''File manipulation utility functions.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from glob import glob
import os.path
import fnmatch
from typing import BinaryIO
from typing import Iterator
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------



#-------------------------------------------------------------------
def readAll(fileName: str) -> str:
  '''Returns all the text from the file.
  '''

  with open(fileName, 'r') as theFile:
    return theFile.read()

#-------------------------------------------------------------------
def readLinesFromEnd(theFile: BinaryIO) -> Iterator[str]:
  '''Generator function that returns the lines in the file, starting at the end.

  @param theFile The file to be read.  Must have been opened with 'b', or
                 be a BytesIO object.

  @return A string that is a single line from the file.
  '''

  theFile.seek(0, os.SEEK_END)

  # The byte in the file we're looking at.
  pointerLocation: int = theFile.tell()

  # The line we're going to next return.
  theBuffer = bytearray()

  while pointerLocation >= 0:

    # Move the file pointer back one.
    theFile.seek(pointerLocation)
    pointerLocation -= 1

    newByte: bytes = theFile.read(1)

    if newByte == b'\n':

      # If we hit a newline character, we've found the start of a line.
      # Decode from utf-8 bytes to a str, reverse it, and yield it.
      yield theBuffer.decode('utf-8')[::-1]

      # Clear the buffer.
      theBuffer = bytearray()

    else:

      # Keep looking for the start of the line.
      theBuffer.extend(newByte)

  # The file has been read completely.
  # Yield the first line.

  if len(theBuffer) > 0:

    # Decode from utf-8 bytes to a str, reverse it, and yield it.
    yield theBuffer.decode('utf-8')[::-1]

#-------------------------------------------------------------------
def recursiveGlob(rootDir: str, pattern: str) -> List[str]:
  '''Look for all files whose name match the pattern, starting in rootDir.
  '''

  found = []
  for dirpath, unusedDirnames, filenames in os.walk(rootDir):
    for f in fnmatch.filter(filenames, pattern):
      found.append(os.path.join(dirpath, f))

  found.sort()
  return found

#-------------------------------------------------------------------
def globEx(pattern: str, maxDepth: int = 10) -> List[str]:
  '''Like glob(), except ! (not) and a single ** are allowed.

  @param pattern A glob()-compatible string, with a single ** allowed.
                 The ** sequence means will match any directories.
                 A bang ! means the directory will be excluded.
  @param maxDepth How many directories the ** can match.

  @return A list of file name strings.
  '''

  if pattern.count('**') > 1:
    msg = "Only 1 ** is allowed per pattern.  Pattern:  " + pattern
    raise ValueError(msg)

  if pattern.count('!') > 1:
    msg = "Only 1 ! is allowed per pattern.  Pattern:  " + pattern
    raise ValueError(msg)

  # If negation is being used, recursively call this function twice,
  # once to find all the files, and once to find the files that
  # shouldn't be returned.

  if '!' in pattern:

    # Find the directory name being negated.
    begin = pattern.index('!')
    end = pattern.find('/', begin + 1)
    if end == -1:
      end = len(pattern)

    # The files that shouldn't be returned.
    negativePattern = pattern.replace('!', '')
    negativeFiles = globEx(negativePattern)

    # The files that should be returned.
    positivePattern = pattern[:begin] + '*' + pattern[end:]
    positiveFiles = globEx(positivePattern)

    for f in negativeFiles:
      positiveFiles.remove(f)

    return positiveFiles

  files = []

  # Handle the ** by expanding into sequences of /*/*
  # and calling glob() with those.

  if '**' not in pattern:
    maxDepth = 1

  for i in range(maxDepth):
    newPattern = pattern.replace("**", "/".join(["*"] * i))
    files.extend(glob(newPattern))

  files.sort()
  return files
