#-------------------------------------------------------------------
#  ArrayUtil.py
#
#  The ArrayUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Utility functions for arrays.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from typing import List
from typing import TypeVar
from typing import Union
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def multiply(array2d, value):
  '''Multiplies every value in a 2-D array and returns a new array.

  @param array2d The array to be multiplied.
  @param value The value to multiply by.

  @return A new 2D array of multiplied values.
  '''

  newArray2d = []
  for row in array2d:
    newRow = [None] * len(row)
    newArray2d.append(newRow)
    for columnNum, column in enumerate(row):
      if column is not None:
        newRow[columnNum] = column * value

  return newArray2d

#-------------------------------------------------------------------
def call(array2d, function, rawNones=False):
  '''Calls the function on every element of the array, returning a new array.

  @param array2d The data that will be passed into the function.
  @param function A function that accepts a single item from the array2d
                  as input, and returns a single item.
  @param rawNones If True, when a None is found in array2D, it will be returned
                    unchanged, rather than being sent into the function.

  @return A 2-D array created by calling the passed-in function with each
          item in the passed-in array.
  '''

  newArray2d = []
  for row in array2d:
    newRow = [None] * len(row)
    newArray2d.append(newRow)
    for columnNum, column in enumerate(row):
      if not rawNones or column is not None:
        newRow[columnNum] = function(column)

  return newArray2d

#-------------------------------------------------------------------
# pylint: disable=C0103
T = TypeVar('T')
# pylint: disable=C0103

#-------------------------------------------------------------------
def grow(array2d: List[List[T]], numRepeats: int) -> List[List[T]]:
  '''Grows an array so that every item in it is repeated numRepeats times.

  @param array2d The array to grow.
  @param numRepeats The number of times to repeat each element.

  @return A new array.
  '''

  newArray2d: List[List[T]] = []
  for row in array2d:

    # Generate the new row list with temp values.
    newRow: List[T] = row[:] * numRepeats

    # Copy the correct values in to the new row.
    for columnNum, column in enumerate(row):
      for i in range(numRepeats):
        newRow[(columnNum * numRepeats) + i] = column

    for i in range(numRepeats):
      newArray2d.append(newRow[:])

  return newArray2d

#-------------------------------------------------------------------
def shrink(array2d: List[List[Union[float, None]]],
           numItems: int) -> List[List[Union[float, None]]]:
  '''Shrinks an array so that every numItems items are shrunk into one item.

  @param array2d The array to shrink.
  @param numItems The number of items that are condensed into one.

  @return A new array.
  '''

  width = len(array2d[0])
  height = len(array2d)

  # Round up by adding (numItems - 1), then doing integer division.
  newWidth = (width + (numItems - 1)) / numItems
  newHeight = (height + (numItems - 1)) / numItems

  newWidth = int(newWidth)
  newHeight = int(newHeight)

  newArray2d: List[List[Union[float, None]]] = []
  for i in range(newHeight):
    newRow: List[Union[float, None]] = [None] * newWidth
    newArray2d.append(newRow)
    for j in range(newWidth):

      # Sum the N x N grid that we're turning into a single value.

      sum: float = 0
      numGridItems = 0
      for iItem in range(numItems):
        for jItem in range(numItems):
          iOld = i * numItems + iItem
          jOld = j * numItems + jItem
          if iOld < height and jOld < width and array2d[iOld][jOld] is not None:
            sum += array2d[iOld][jOld] # type: ignore
            numGridItems += 1

      if numGridItems:

        # Find the average.
        average: float = float(sum) / numGridItems

        newArray2d[i][j] = average

  return newArray2d

#-------------------------------------------------------------------
def toStr(array2d: List[List[object]]) -> str:
  '''Formats a 2-D array into a string.
  '''

  rowStrs = []

  for row in array2d:
    rowStrs.append(str(row))

  return '[' + ',\n'.join(rowStrs) + ']'
