#-------------------------------------------------------------------
#  TestAngle.py
#
#  The TestAngle module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the Angle class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonDecoder
from ai.axe.classJson import ClassJsonEncoder
from ai.axe.math import Angle
from ai.axe.build.unittest import AxeSimpleTestCase
import math
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestAngle(AxeSimpleTestCase):
  '''Unit test for the Angle class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testTestBasic(self) -> None:
    '''Test basic angle functions.
    '''

    angle = Angle.fromDegrees(0)
    self.assertClose(0, angle.radians, 'degrees -> radians')

    angle = Angle.fromRadians(0)
    self.assertClose(0, angle.degrees, 'radians -> degrees')

    angle = Angle.fromDegrees(180)
    self.assertClose(math.pi, angle.radians, 'degrees -> radians')

    angle = Angle.fromRadians(math.pi)
    self.assertClose(180, angle.degrees, 'radians -> degrees')

    angle = Angle.fromDegrees(360)
    self.assertClose(0, angle.degrees, 'normalized degrees')
    self.assertClose(0, angle.radians, 'normalized radians')

    angle = Angle.fromRadians(2 * math.pi)
    self.assertClose(0, angle.degrees, 'normalized degrees')
    self.assertClose(0, angle.radians, 'normalized radians')

    angle = Angle.fromRadians(-4.5 * math.pi)
    self.assertClose(270, angle.degrees, 'normalized degrees')
    self.assertClose(1.5 * math.pi, angle.radians, 'normalized radians')

    angle = Angle.fromDegrees(90).plus(Angle.fromDegrees(15))
    self.assertClose(105, angle.degrees, 'plus')

    angle = Angle.fromDegrees(90) + Angle.fromDegrees(15)
    self.assertClose(105, angle.degrees, '__add__')

    angle = Angle.fromDegrees(90)
    angle += Angle.fromDegrees(15)
    self.assertClose(105, angle.degrees, '__iadd__')

    angle = Angle.fromRadians(math.pi).minus(Angle.fromDegrees(45))
    self.assertClose(135, angle.degrees, 'minus')

    angle = Angle.fromRadians(math.pi) - Angle.fromDegrees(45)
    self.assertClose(135, angle.degrees, '__sub__')

    angle = Angle.fromRadians(math.pi)
    angle -= Angle.fromDegrees(45)
    self.assertClose(135, angle.degrees, '__isub__')

    angle = Angle.fromDegrees(90).times(2)
    self.assertClose(180, angle.degrees, 'times')

    angle = Angle.fromDegrees(90) * 2
    self.assertClose(180, angle.degrees, '__mul_')

    angle = 2 * Angle.fromDegrees(90)
    self.assertClose(180, angle.degrees, '__rmul_')

    angle = Angle.fromDegrees(90)
    angle *= 2
    self.assertClose(180, angle.degrees, '__imul_')

    angle = Angle.fromDegrees(91).dividedBy(2)
    self.assertClose(45.5, angle.degrees, 'dividedBy')

    angle = Angle.fromDegrees(91) / 2
    self.assertClose(45.5, angle.degrees, '__truediv__')

    angle = Angle.fromDegrees(91)
    angle /= 2
    self.assertClose(45.5, angle.degrees, '__itruediv__')

    angle = -Angle.fromDegrees(90)
    self.assertClose(270, angle.degrees, '__neg_')

    self.assertClose(1, Angle.fromDegrees(0).cos(), 'cos 0')
    self.assertClose(-1, Angle.fromDegrees(180).cos(), 'cos 180')
    self.assertClose(0, Angle.fromDegrees(270).cos(), 'cos 270')

    self.assertClose(0, Angle.fromDegrees(0).sin(), 'sin 0')
    self.assertClose(1, Angle.fromDegrees(90).sin(), 'sin 90')
    self.assertClose(0, Angle.fromDegrees(180).sin(), 'sin 180')
    self.assertClose(-1, Angle.fromDegrees(270).sin(), 'cos 270')

    self.assertClose(90, Angle.fromDegrees(0).mathToMapAngle().degrees, 'm2m 0')
    self.assertClose(0, Angle.fromDegrees(90).mapToMathAngle().degrees, 'm2m 0')

    # Test that str() doesn't raise an exception and raise coverage.
    str(angle)

  #-----------------------------------------------------------------
  def testCompare(self) -> None:
    '''Test comparing with == and !=.
    '''

    self.assertEqual(True, Angle.fromDegrees(90) == Angle.fromDegrees(90), 'eq')
    self.assertEqual(False, Angle.fromDegrees(90) == Angle.fromDegrees(91),
                     'eq')
    self.assertEqual(True, Angle.fromDegrees(90) != Angle.fromDegrees(91), 'ne')
    self.assertEqual(False, Angle.fromDegrees(90) != Angle.fromDegrees(90),
                     'ne')

    self.assertEqual(hash(Angle.fromDegrees(90)),
                     hash(Angle.fromDegrees(90)),
                     'hash')
    self.assertNotEqual(hash(Angle.fromDegrees(90)),
                        hash(Angle.fromDegrees(91)),
                        'hash')

  #-----------------------------------------------------------------
  def testJson(self) -> None:
    '''Test converting between JSON and Python.
    '''

    jsonStr = ClassJsonEncoder().encode(Angle.fromRadians(math.pi))
    angle = ClassJsonDecoder().decode(jsonStr, Angle)

    self.assertClose(180, angle.degrees, 'json')

  #-----------------------------------------------------------------
  def assertClose(self, expected, actual, msg) -> None:

    self.assertAlmostEqual(expected, actual, places=12, msg=msg)
