#-------------------------------------------------------------------
#  Rectangle.py
#
#  The Rectangle class.
#
#  Copyright 2019 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the Rectangle class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.util import StringUtil
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
@ClassJsonClass([ClassJsonField('x', float),
                 ClassJsonField('y', float),
                 ClassJsonField('width', float),
                 ClassJsonField('height', float)])
class Rectangle:
  '''A rectangle, comprising an (x, y) origin and (width, height) size.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               x: float,
               y: float,
               width: float,
               height: float) -> None:
    '''Creates a new Rectangle.
    '''

    # The x value of the location.
    self.x: float = x

    # The y value of the location.
    self.y: float = y

    # The width of the rectangle.
    self.width: float = width

    # The height of the rectangle.
    self.height: float = height

  #-----------------------------------------------------------------
  def key(self) -> Tuple[float, float, float, float]:
    '''Returns a value that can be used as a dict key.
    '''

    return self.x, self.y, self.width, self.height

  #-----------------------------------------------------------------
  def __eq__(self, other: object) -> bool:

    if not isinstance(other, Rectangle):
      return False

    return self.key() == other.key()

  #-----------------------------------------------------------------
  def __ne__(self, other: object) -> bool:

    return not self.__eq__(other)

  #-----------------------------------------------------------------
  def __hash__(self):

    return hash(self.key())

  #-----------------------------------------------------------------
  @staticmethod
  def fromPostgresPoints(upperRight: Tuple[float, float],
                         lowerLeft: Tuple[float, float]) -> 'Rectangle':
    '''Creates a new Rectangle object from upperRight/lowerLeft values.

    This is the format in which Postgres stores rectangles internally.

    @param upperRight The upper-right x/y coordinates.
    @param lowerLeft The lower-left x/y coordinates.

    @return A new Rectangle object.
    '''

    return Rectangle(lowerLeft[0],
                     lowerLeft[1],
                     upperRight[0] - lowerLeft[0],
                     upperRight[1] - lowerLeft[1])

  #-----------------------------------------------------------------
  def toPostgresPoints(self) -> Tuple[Tuple[float, float],
                                      Tuple[float, float]]:
    '''Returns upperRight/lowerLeft values.

    This is the format in which Postgres stores rectangles internally.

    @return An upperRight, lowerLeft tuple.
    '''

    upperRight = (self.x + self.width), (self.y + self.height)
    return upperRight, (self.x, self.y)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['x', 'y', 'width', 'height']

    return StringUtil.formatRepr(self, attrs)
