#-------------------------------------------------------------------
#  Angle.py
#
#  The Angle class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the Angle class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.util import StringUtil
import math
#
# Import statements go above this line.
#-------------------------------------------------------------------

# Allow same-name degree and radian data member and static constructor.
# pylint: disable=E0202

#===================================================================
@ClassJsonClass([ClassJsonField('degrees', float)],
                ctor='fromDegrees')
class Angle:
  '''A geometric angle.

  You can create an Angle from a number of degrees of radians
  using static constructor functions like so:

    angle = Angle.degrees(180)
    angle = Angle.radians(2 * math.pi)

  Once created, you can access the value in degress or radians:

     numRadians = angle.radians
     numDegrees = angle.degrees
  '''

  #-----------------------------------------------------------------
  radiansPerDegree: float = math.pi / 180

  #-----------------------------------------------------------------
  degreesPerRadian: float = 180 / math.pi

  #-----------------------------------------------------------------
  def __init__(self, degrees: float, radians: float) -> None:
    '''Creates a new Angle.
    '''

    # This angle in degrees.
    self.degrees: float = degrees

    # This angle in radians.
    self.radians: float = radians

  #-----------------------------------------------------------------
  @staticmethod
  def fromDegrees(degrees: float) -> 'Angle':
    '''Creates a new Angle object that is the specified number of degrees.

    @param degrees The number of degrees to create the new Angle with.

    @return A new Angle object.
    '''

    degrees = Angle.normalizedDegrees(degrees)
    radians = degrees * Angle.radiansPerDegree

    return Angle(degrees, radians)

  #-----------------------------------------------------------------
  @staticmethod
  def fromRadians(radians: float) -> 'Angle':
    '''Creates a new Angle object that is the specified number of radians.

    @param radians The number of radians to create the new Angle with.

    @return A new Angle object.
    '''

    radians = Angle.normalizedRadians(radians)
    degrees = radians * Angle.degreesPerRadian

    return Angle(degrees, radians)

  #-----------------------------------------------------------------
  @staticmethod
  def normalizedDegrees(degrees: float) -> float:
    '''Returns an equivalent number of degrees that is 0 <= x < 360.

    @param degrees The number of degrees to be normalized.

    @return The normalized number of degrees.
    '''

    return Angle.normalized(degrees, 360)

  #-----------------------------------------------------------------
  @staticmethod
  def normalizedRadians(radians: float) -> float:
    '''Returns an equivalent number of radians that is 0 <= x < 2 * PI.

    @param radians The number of radians to be normalized.

    @return The normalized number of radians.
    '''

    return Angle.normalized(radians, 2 * math.pi)

  #-----------------------------------------------------------------
  @staticmethod
  def normalized(n: float, maxAmount: float) -> float:
    '''Returns an equivalent number that is 0 <= x < maxAmount.

    @param n The number to be normalized.

    @return The normalized number.
    '''

    return n % maxAmount

  #-----------------------------------------------------------------
  def __eq__(self, other: object) -> bool:

    if not isinstance(other, Angle):
      return False

    return self.degrees == other.degrees

  #-----------------------------------------------------------------
  def __ne__(self, other: object) -> bool:

    return not self.__eq__(other)

  #-----------------------------------------------------------------
  def __hash__(self):

    return hash(self.degrees)

  #-----------------------------------------------------------------
  def __add__(self, other: 'Angle') -> 'Angle':

    return self.plus(other)

  #-----------------------------------------------------------------
  def __iadd__(self, other: 'Angle') -> 'Angle':

    return self.plus(other)

  #-----------------------------------------------------------------
  def __sub__(self, other: 'Angle') -> 'Angle':

    return self.minus(other)

  #-----------------------------------------------------------------
  def __isub__(self, other: 'Angle') -> 'Angle':

    return self.minus(other)

  #-----------------------------------------------------------------
  def __mul__(self, other: float) -> 'Angle':

    return self.times(other)

  #-----------------------------------------------------------------
  def __rmul__(self, other: float) -> 'Angle':

    return self.times(other)

  #-----------------------------------------------------------------
  def __imul__(self, other: float) -> 'Angle':

    return self.times(other)

  #-----------------------------------------------------------------
  def __truediv__(self, other: float) -> 'Angle':

    return self.dividedBy(other)

  #-----------------------------------------------------------------
  def __itruediv__(self, other: float) -> 'Angle':

    return self.dividedBy(other)

  #-----------------------------------------------------------------
  def __neg__(self) -> 'Angle':

    return Angle.fromDegrees(-self.degrees)

  #-----------------------------------------------------------------
  def plus(self, otherAngle: 'Angle') -> 'Angle':
    '''Adds this angle to another angle.

    @param otherAngle The other angle to be added.

    @return A new Angle object that is the sum of the two angles.
    '''

    newDegrees = Angle.normalizedDegrees(self.degrees + otherAngle.degrees)
    return Angle.fromDegrees(newDegrees)

  #-----------------------------------------------------------------
  def minus(self, otherAngle: 'Angle') -> 'Angle':
    '''Subtracts another angle from this one.

    @param otherAngle The other angle to be subtracted.

    @return A new Angle object that is the difference of the two angles.
    '''

    newDegrees = Angle.normalizedDegrees(self.degrees - otherAngle.degrees)
    return Angle.fromDegrees(newDegrees)

  #-----------------------------------------------------------------
  def times(self, n: float) -> 'Angle':
    '''Multiplies this angle by a number n.

    @param n The number to multiply this angle by.

    @return A new Angle object that is the product of the two numbers.
    '''

    newDegrees = Angle.normalizedDegrees(self.degrees * n)
    return Angle.fromDegrees(newDegrees)

  #-----------------------------------------------------------------
  def dividedBy(self, n: float) -> 'Angle':
    '''Divides this angle by a number n.

    @param n The number to divide this angle by.

    @return A new Angle object that is the qotient of the two numbers.
    '''

    newDegrees = Angle.normalizedDegrees(self.degrees / n)
    return Angle.fromDegrees(newDegrees)

  #-----------------------------------------------------------------
  def cos(self) -> float:
    '''Returns the cosine of this angle.
    '''

    return math.cos(self.radians)

  #-----------------------------------------------------------------
  def sin(self) -> float:
    '''Returns the sine of this angle.
    '''

    return math.sin(self.radians)

  #-----------------------------------------------------------------
  def mathToMapAngle(self) -> 'Angle':
    '''Converts this 'math' angle to a 'map' angle.

    A 'math' angle starts at 3 o'clock and increases counterclockwise.
    A 'map' angle starts at 12 o'clock and increases clockwise.

    @return A new Angle object which is the equivalent 'map' angle.
    '''

    return Angle.fromDegrees(-self.degrees + 90.0)

  #-----------------------------------------------------------------
  def mapToMathAngle(self) -> 'Angle':
    '''Converts this 'map' angle to a 'math' angle.

    A 'math' angle starts at 3 o'clock and increases counterclockwise.
    A 'map' angle starts at 12 o'clock and increases clockwise.

    @return A new Angle object which is the equivalent 'math' angle.
    '''

    return Angle.fromDegrees(-self.degrees + 90.0)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['degrees', 'radians']

    return StringUtil.formatRepr(self, attrs)
