#-------------------------------------------------------------------
#  DayFileHandler.py
#
#  The DayFileHandler class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the DayFileHandler class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import FileUtil
from ai.axe.util import StringUtil
from datetime import date
from datetime import datetime
from logging import FileHandler
from logging import LogRecord
from typing import Optional
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class DayFileHandler(FileHandler):
  '''A file handler that creates a new log file every day.

  This works around an issue with TimedRotatingFileHandler in Python 3.7.
  TimedRoatingFileHandler checks the file modification time on startup,
  and rotates the file only if the last modification time is over a day ago.
  It next rotates the file one day after startup.  So if you have
  a Python program running for 10 minutes once per hour, the file never
  gets rotated, since the file modification time is never one day old,
  and the program never runs for a day.

  DayFileHandler works like so.  Each message gets written to a file
  BASENAME.YYYY-MM-DD.  There is also a symlink BASNAME which is always
  set to point to the newest log file.

  This file/symlink avoids issues with messages going strage places during
  a rolling file copy or move.
  '''

  #-----------------------------------------------------------------
  def __init__(self, fileName: str) -> None:
    '''Creates a new DayFileHandler.

    @param fileName The base file name to write.
    '''

    # The file name of the log file.
    # The symlink will have this name.
    self.fileNameBase = fileName

    # The file name pattern.
    # Actual file names are generated with strftime.
    self.fileNamePattern = fileName + '.%Y-%m-%d'

    fileName = datetime.utcnow().strftime(self.fileNamePattern)
    mode: str = 'a'
    encoding: Optional[str] = None
    delay: bool = False

    super().__init__(fileName, mode, encoding, delay)

    self.updateSymLink(fileName)

  #-----------------------------------------------------------------
  def emit(self, record: LogRecord):

    createdAt: datetime = datetime.fromtimestamp(record.created)
    newFileName = createdAt.strftime(self.fileNamePattern)

    if self.stream is None:
      self.setNewFileName(newFileName)

    elif self.differsFromCurrentFilename(newFileName):
      self.close()
      self.setNewFileName(newFileName)

    super().emit(record)

    self.updateSymLink(newFileName)

  #-----------------------------------------------------------------
  def setNewFileName(self, newFileName):
    '''Sets the FileHandler.baseFilename that future emit() calls will write to.
    '''

    self.baseFilename = newFileName

  #-----------------------------------------------------------------
  def differsFromCurrentFilename(self, fileName: str) -> bool:
    '''True if the specified file name is not what we've been emitting to.
    '''

    return fileName != self.baseFilename

  #----------------------------------------------------------------
  def updateSymLink(self, fileName: str) -> None:
    '''Updates the file symlink to point to the the file we're writing to.
    '''

    if (os.path.isfile(self.fileNameBase) and
        not os.path.islink(self.fileNameBase)):
      self.handleExistingFile()

    if os.path.islink(self.fileNameBase):

      target: str = os.readlink(self.fileNameBase)

      if target != fileName:

        os.unlink(self.fileNameBase)

    if not os.path.exists(self.fileNameBase):

      os.symlink(fileName, self.fileNameBase)

#        On linux, symlinks have the same permissions as the file
#        they point to.  If we want to support multiple users running
#        commands on a single install on MacOS, we'd have to change the
#        symlink permissions like so.
#
#        mode = (stat.S_IRUSR | stat.S_IWUSR |
#                stat.S_IRGRP | stat.S_IWGRP |
#                stat.S_IROTH)
#
#        os.lchmod(self.fileNameBase, mode)

  #----------------------------------------------------------------
  def handleExistingFile(self) -> None:
    '''The file we expected to be a symlink is an actual log file.  Handle it.
    '''

    # Find the date of the last message.

    theDate: Optional[date] = None
    theDateStr: Optional[str] = None

    for line in FileUtil.readLinesFromEnd(open(self.fileNameBase, 'rb')):

      if len(line) < 10:
        continue

      try:
        theDate = datetime.strptime(line[:10], '%Y-%m-%d').date()

      except ValueError:
        continue

      assert theDate is not None
      theDateStr = line[:10]

      break

    # Try to rename the file to *.YYYY-MM-DD.
    # If that doesn't work, try *.old
    # If that doesn't work, try *.old.1 and so on.

    if theDateStr is not None:

      newName = self.fileNameBase + '.' + theDateStr
      if not os.path.exists(newName):
        os.rename(self.fileNameBase, newName)
        return

    newName = self.fileNameBase + '.old'
    if not os.path.exists(newName):
      os.rename(self.fileNameBase, newName)
      return

    for i in range(100):

      newName = self.fileNameBase + ('.old.%s' % i)
      if not os.path.exists(newName):
        os.rename(self.fileNameBase, newName)
        return

    msg = "Failed to rename existing log file: %s" % self.fileNameBase
    raise ValueError(msg)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['fileNameBase']

    return StringUtil.formatRepr(self, attrs)
