#-------------------------------------------------------------------
#  TestParseJsComment.py
#
#  The TestParseJsComment module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the ParseJsComment class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.js import ParseJsComment
from ai.axe.build.unittest import AxeSimpleTestCase
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestParseJsComment(AxeSimpleTestCase):
  '''Unit test for the ParseJsComment class.
'''

  #-----------------------------------------------------------------
  def setUp(self):

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self):

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testBasic(self):
    '''Find simple comments.
    '''

    text = '''/* Comment 1.
 *
 */

var foo = bar;

// Single-line comment.
'''

    expected = [(0, 20), (38, 62)]
    actual = ParseJsComment.findComments(text)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testShort(self):
    '''Find short comments.
    '''

    text = '''/**/

var foo = bar;

//'''

    expected = [(0, 4), (22, 24)]
    actual = ParseJsComment.findComments(text)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testCommentInString(self):
    '''Find comments mixed with strings.
    '''

    text = r'''/**/ "/*\"*/" /**/ '/**/' \n /**/'''
    expected = [(0, 4), (14, 18), (29, 33)]
    actual = ParseJsComment.findComments(text)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testStringInComment(self):
    '''Find comments mixed with strings.
    '''

    text = '''/* "Hello *//* there" */ "//"  /* " // " */'''

    expected = [(0, 12), (12, 24), (31, 43)]
    actual = ParseJsComment.findComments(text)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testUnfinished(self):
    '''Test unfinished strings and comments.
    '''

    try:
      text = ''' "Hello '''
      ParseJsComment.findComments(text)
      self.fail("No exception for unfinished string.")
    except ValueError:
      pass

    try:
      text = ''' /* '''
      ParseJsComment.findComments(text)
      self.fail("No exception for unfinished comment.")
    except ValueError:
      pass

  #-----------------------------------------------------------------
  def testRegex(self):
    '''Handle quotes in regex literals.
    '''

    text = '''
var x = /a.*/g;
// Comment.
var y = /^["]/g;
'''

    expected = [(17, 29)]
    actual = ParseJsComment.findComments(text)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testRegexComments(self):
    '''Handle regex literals and comments on the same line.
    '''

    text = r'''
var x = /a.*/g; /* Comment.
*/
var y = /^["]/g; // Comment.
var y = /^[//]/g; // Comment.
var y = /^\//g;
'''

    expected = [(17, 31), (49, 61), (79, 91)]
    actual = ParseJsComment.findComments(text)

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testFindRegexEnd(self):
    '''Test finding the end of a regular expression.
    '''

    # Valid.

    text = r'/abc/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc [/]de/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc\/de/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc [\/]de/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc [\\/]de/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc [\\]de/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc ][\\]de/'
    expected = len(text) - 1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)


    # Invalid.

    text = r'/abc'
    expected = -1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = '''/abc
 3 / 5'''
    expected = -1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc [\]de/'
    expected = -1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)

    text = r'/abc [\\\]de/'
    expected = -1
    actual = ParseJsComment.findRegexEnd(text, 0)
    self.assertEqual(expected, actual)
