#-------------------------------------------------------------------
#  TestSourceNode.py
#
#  The TestSourceNode module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the SourceNode class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.js.jsSourceMap import SourceMapException
from ai.axe.js.jsSourceMap import LineColumn
from ai.axe.js.jsSourceMap import SourceMapConsumer
from ai.axe.js.jsSourceMap import SourceMapGenerator
from ai.axe.js.jsSourceMap import SourceNode
from ai.axe.build.unittest import AxeSimpleTestCase
from . import SampleData
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestSourceNode(AxeSimpleTestCase):
  '''Unit test for the SourceNode class.
'''

  #-----------------------------------------------------------------
  def setUp(self):

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self):

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testAdd(self):

    node = SourceNode(None, None, None)

    # Adding a string works.
    node.add('function noop() {}')

    # Adding another source node works.
    node.add(SourceNode(None, None, None))

    # Adding an array works.
    node.add(['function foo() {',
              SourceNode(None, None, None,
                             'return 10;'),
              '}'])

    # Adding other stuff doesn't.

    try:
      node.add({})
      self.fail("no exception for map")
    except SourceMapException:
      pass

    try:
      node.add(lambda x: x)
      self.fail("no exception for function")
    except SourceMapException:
      pass

  #-----------------------------------------------------------------
  def testPrepend(self):

    node = SourceNode(None, None, None)

    # Prepending a string works.
    node.prepend('function noop() {}')
    self.assertEqual(node.children[0], 'function noop() {}')
    self.assertEqual(len(node.children), 1)

    # Prepending another source node works.
    node.prepend(SourceNode(None, None, None))
    self.assertEqual(node.children[0].toString(), '')
    self.assertEqual(node.children[1], 'function noop() {}')
    self.assertEqual(len(node.children), 2)

    # Prepending an array works.
    node.prepend(['function foo() {',
                  SourceNode(None, None, None,
                             'return 10;'),
                  '}'])
    self.assertEqual(node.children[0], 'function foo() {')
    self.assertEqual(node.children[1].toString(), 'return 10;')
    self.assertEqual(node.children[2], '}')
    self.assertEqual(node.children[3].toString(), '')
    self.assertEqual(node.children[4], 'function noop() {}')
    self.assertEqual(len(node.children), 5)

    # Prepending other stuff doesn't.

    try:
      node.prepend({})
      self.fail("no exception for map")
    except SourceMapException:
      pass


    try:
      node.prepend(lambda x: x)
      self.fail("no exception for function")
    except SourceMapException:
      pass

  #-----------------------------------------------------------------
  def testToString(self):

    node = SourceNode(None, None, None,
                      ['function foo() {',
                       SourceNode(None, None, None, 'return 10;'),
                       '}'])
    self.assertEqual(node.toString(),
                     'function foo() {return 10;}')

  #-----------------------------------------------------------------
  def testJoin(self):

    node = SourceNode(None, None, None,
                      ['a', 'b', 'c', 'd'])
    self.assertEqual(node.join(', ').toString(),
                     'a, b, c, d')

  #-----------------------------------------------------------------
  def testWalk(self):

    children = ['(function () {\n',
                '  ', SourceNode(1, 0, 'a.js', ['someCall()']), ';\n',
                '  ', SourceNode(2, 0, 'b.js', ['if (foo) bar()']), ';\n',
                '}());']
    node = SourceNode(None, None, None, children)

    expected = [
      {'str': '(function () {\n', 'source': None, 'line': None, 'column': None},
      {'str': '  ', 'source': None, 'line': None, 'column': None},
      {'str': 'someCall()', 'source': 'a.js', 'line': 1, 'column': 0},
      {'str': ';\n', 'source': None, 'line': None, 'column': None},
      {'str': '  ', 'source': None, 'line': None, 'column': None},
      {'str': 'if (foo) bar()', 'source': 'b.js', 'line': 2, 'column': 0},
      {'str': ';\n', 'source': None, 'line': None, 'column': None},
      {'str': '}());', 'source': None, 'line': None, 'column': None},
    ]

    asserter = self

    class Checker:
      def __init__(self):
        self.i = 0

      def check(self, chunk, loc):
        asserter.assertEqual(expected[self.i]["str"], chunk)
        asserter.assertEqual(expected[self.i]["source"], loc.source)
        asserter.assertEqual(expected[self.i]["line"], loc.line)
        asserter.assertEqual(expected[self.i]["column"], loc.column)
        self.i += 1

      __call__ = check

    node.walk(Checker())

  #-----------------------------------------------------------------
  def testReplaceRight(self):

    # Not nested
    node = SourceNode(None, None, None, 'hello world')
    node.replaceRight('world', 'universe')
    self.assertEqual(node.toString(), 'hello universe')

    # Nested
    node = SourceNode(None, None, None,
                      [SourceNode(None, None, None, 'hey sexy mama, '),
                      SourceNode(None, None, None, 'want to kill all humans?')])
    node.replaceRight('kill all humans', 'watch Futurama')
    self.assertEqual(node.toString(), 'hey sexy mama, want to watch Futurama?')

  #-----------------------------------------------------------------
  def testToStringWithSourceMap(self):

    node = SourceNode(None, None, None,
                      ['(function () {\n',
                       '  ',
                       SourceNode(1, 0, 'a.js', 'someCall', 'originalCall'),
                       SourceNode(1, 8, 'a.js', '()'),
                       ';\n',
                      '  ', SourceNode(2, 0, 'b.js', ['if (foo) bar()']), ';\n',
                       '}());'])
    dummyCode, theMap = node.toStringWithSourceMap(file='foo.js')

    self.assertTrue(isinstance(theMap, SourceMapGenerator),
                    'map instanceof SourceMapGenerator')
    theMap = SourceMapConsumer(theMap.toJson())

    actual = theMap.originalPositionFor(LineColumn(1, 4))
    self.assertEqual(actual.source, None)
    self.assertEqual(actual.line, None)
    self.assertEqual(actual.column, None)

    actual = theMap.originalPositionFor(LineColumn(2, 2))
    self.assertEqual(actual.source, 'a.js')
    self.assertEqual(actual.line, 1)
    self.assertEqual(actual.column, 0)
    self.assertEqual(actual.name, 'originalCall')

    actual = theMap.originalPositionFor(LineColumn(3, 2))
    self.assertEqual(actual.source, 'b.js')
    self.assertEqual(actual.line, 2)
    self.assertEqual(actual.column, 0)

    actual = theMap.originalPositionFor(LineColumn(3, 16))
    self.assertEqual(actual.source, None)
    self.assertEqual(actual.line, None)
    self.assertEqual(actual.column, None)

    actual = theMap.originalPositionFor(LineColumn(4, 2))
    self.assertEqual(actual.source, None)
    self.assertEqual(actual.line, None)
    self.assertEqual(actual.column, None)

  #-----------------------------------------------------------------
  def testFromStringWithSourceMap(self):

    node = SourceNode.fromStringWithSourceMap(
                              SampleData.generatedCode,
                              SourceMapConsumer(SampleData.mapData))

    code, theMap = node.toStringWithSourceMap(file='min.js')

    self.assertEqual(code, SampleData.generatedCode)
    self.assertTrue(isinstance(theMap, SourceMapGenerator),
                    'map instanceof SourceMapGenerator')

    theMap = theMap.toJson()
    self.assertEqual(theMap['version'], SampleData.mapData['version'])
    self.assertEqual(theMap['file'], SampleData.mapData['file'])
    self.assertEqual(theMap['mappings'], SampleData.mapData['mappings'])

  #-----------------------------------------------------------------
  def testFromStringWithEmptySourceMap(self):

    node = SourceNode.fromStringWithSourceMap(SampleData.generatedCode,
                          SourceMapConsumer(SampleData.emptyMap))
    code, theMap = node.toStringWithSourceMap(file='min.js')

    self.assertEqual(code, SampleData.generatedCode)
    self.assertTrue(isinstance(theMap, SourceMapGenerator),
                    'map instanceof SourceMapGenerator')

    theMap = theMap.toJson()
    self.assertEqual(theMap['version'], SampleData.emptyMap['version'])
    self.assertEqual(theMap['file'], SampleData.emptyMap['file'])
    self.assertEqual(len(theMap['mappings']),
                     len(SampleData.emptyMap['mappings']))
    self.assertEqual(theMap['mappings'], SampleData.emptyMap['mappings'])

  #-----------------------------------------------------------------
  def testFromStringWithSourceMapComplextVersion(self):

    inputNode = SourceNode(None, None, None, [
      "(function() {\n",
        "  var Test = {};\n",
        "  ", SourceNode(1, 0, "a.js", "Test.A = { value: 1234 };\n"),
        "  ", SourceNode(2, 0, "a.js", "Test.A.x = 'xyz';"), "\n",
        "}());\n",
        "/* Generated Source */"])
    inputCode, inputMap = inputNode.toStringWithSourceMap(file='foo.js')

    node = SourceNode.fromStringWithSourceMap(
                              inputCode,
                              SourceMapConsumer(inputMap.toJson()))

    code, theMap = node.toStringWithSourceMap(file='foo.js')
    self.assertEqual(code, inputCode)
    self.assertTrue(isinstance(theMap, SourceMapGenerator),
                    'map instanceof SourceMapGenerator')

    theMap = theMap.toJson()
    inputMap = inputMap.toJson()

    self.assertEqualMaps(theMap, inputMap)

  #-----------------------------------------------------------------
  def testFromStringWithSourceMapMergingDuplicateMappings(self):

    inputNode = SourceNode(None, None, None, [
      SourceNode(1, 0, "a.js", "(function"),
      SourceNode(1, 0, "a.js", "() {\n"),
      "  ",
      SourceNode(1, 0, "a.js", "var Test = "),
      SourceNode(1, 0, "b.js", "{};\n"),
      SourceNode(2, 0, "b.js", "Test"),
      SourceNode(2, 0, "b.js", ".A", "A"),
      SourceNode(2, 20, "b.js", " = { value: 1234 };\n", "A"),
      "}());\n",
      "/* Generated Source */"
    ])
    dummyInputCode, inputMap = inputNode.toStringWithSourceMap(file='foo.js')

    correctMap = SourceMapGenerator(file='foo.js')

    correctMap.addMapping(generated=LineColumn(1, 0),
                          source='a.js',
                          original=LineColumn(1, 0))
    correctMap.addMapping(generated=LineColumn(2, 0))
    correctMap.addMapping(generated=LineColumn(2, 2),
                          source='a.js',
                          original=LineColumn(1, 0))
    correctMap.addMapping(generated=LineColumn(2, 13),
                          source='b.js',
                          original=LineColumn(1, 0))
    correctMap.addMapping(generated=LineColumn(3, 0),
                          source='b.js',
                          original=LineColumn(2, 0))
    correctMap.addMapping(generated=LineColumn(3, 4),
                          source='b.js',
                          name='A',
                          original=LineColumn(2, 0))
    correctMap.addMapping(generated=LineColumn(3, 6),
                          source='b.js',
                          name='A',
                          original=LineColumn(2, 20))
    correctMap.addMapping(generated=LineColumn(4, 0))

    inputMap = inputMap.toJson()
    correctMap = correctMap.toJson()
    self.assertEqualMaps(correctMap, inputMap)

  #-----------------------------------------------------------------
  def testSetSourceContentWithToStringWithSourceMap(self):

    aNode = SourceNode(1, 1, 'a.js', 'a')
    aNode.setSourceContent('a.js', 'someContent')
    node = SourceNode(None, None, None,
                      ['(function () {\n',
                       '  ', aNode,
                       '  ', SourceNode(1, 1, 'b.js', 'b'),
                       '}());'])
    node.setSourceContent('b.js', 'otherContent')
    dummyCode, theMap = node.toStringWithSourceMap(file='foo.js')

    self.assertTrue(isinstance(theMap, SourceMapGenerator),
                    'map instanceof SourceMapGenerator')
    theMap = SourceMapConsumer(theMap.toJson())

    self.assertEqual(len(theMap.sources), 2)
    self.assertEqual(theMap.sources[0], 'a.js')
    self.assertEqual(theMap.sources[1], 'b.js')
    self.assertEqual(len(theMap.sourcesContent), 2)
    self.assertEqual(theMap.sourcesContent[0], 'someContent')
    self.assertEqual(theMap.sourcesContent[1], 'otherContent')

  #-----------------------------------------------------------------
  def testWalkSourceContents(self):

    aNode = SourceNode(1, 1, 'a.js', 'a')
    aNode.setSourceContent('a.js', 'someContent')
    node = SourceNode(None, None, None,
                      ['(function () {\n',
                       '  ', aNode,
                       '  ', SourceNode(1, 1, 'b.js', 'b'),
                       '}());'])
    node.setSourceContent('b.js', 'otherContent')

    class ResultsCollector:
      def __init__(self):
        self.results = []

      def __call__(self, sourceFile, sourceContent):
        self.results.append((sourceFile, sourceContent))

    resultsCollector = ResultsCollector()
    node.walkSourceContents(resultsCollector)
    results = resultsCollector.results

    self.assertEqual(len(results), 2)
    self.assertEqual(results[0][0], 'a.js')
    self.assertEqual(results[0][1], 'someContent')
    self.assertEqual(results[1][0], 'b.js')
    self.assertEqual(results[1][1], 'otherContent')


  #-----------------------------------------------------------------
  assertEqualMaps = SampleData.assertEqualMaps
