#-------------------------------------------------------------------
#  TestSourceMapGenerator.py
#
#  The TestSourceMapGenerator module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the SourceMapGenerator class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.js.jsSourceMap import SourceMapException
from ai.axe.js.jsSourceMap import LineColumn
from ai.axe.js.jsSourceMap import SourceMapConsumer
from ai.axe.js.jsSourceMap import SourceMapGenerator
from ai.axe.js.jsSourceMap import SourceNode
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.js.jsSourceMap.test import SampleData
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestSourceMapGenerator(AxeSimpleTestCase):
  '''Unit test for the SourceMapGenerator class.
'''

  #-----------------------------------------------------------------
  def setUp(self):

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self):

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testSomeSimpleStuff(self):

    SourceMapGenerator(file='foo.js', sourceRoot='.')

  #-----------------------------------------------------------------
  def testToJson(self):

    theMap = SourceMapGenerator(file='foo.js', sourceRoot='.')
    theMap.toJson()

  #-----------------------------------------------------------------
  def testAddMappings1(self):

    theMap = SourceMapGenerator(file='generated-foo.js', sourceRoot='.')
    theMap.addMapping(generated=LineColumn(1, 1))

  #-----------------------------------------------------------------
  def testAddMappings2(self):

    theMap = SourceMapGenerator(file='generated-foo.js', sourceRoot='.')
    theMap.addMapping(generated=LineColumn(1, 1),
                      source='bar.js',
                      original=LineColumn(1, 1))

  #-----------------------------------------------------------------
  def testAddMappings3(self):

    theMap = SourceMapGenerator(file='generated-foo.js', sourceRoot='.')
    theMap.addMapping(generated=LineColumn(1, 1),
                      source='bar.js',
                      original=LineColumn(1, 1),
                      name='someToken')

  #-----------------------------------------------------------------
  def testAddMappingsInvalid(self):

    theMap = SourceMapGenerator(file='generated-foo.js', sourceRoot='.')

    try:
      theMap.addMapping(generated=LineColumn(1, 1),
                        original=LineColumn(1, 1))
      self.fail("No exception thrown.")
    except SourceMapException:
      pass

  #-----------------------------------------------------------------
  def testCorrectMappingsBeingGenerated(self):

    theMap = SourceMapGenerator(file='min.js', sourceRoot='/the/root')

    theMap.addMapping(generated=LineColumn(1, 1),
                      original=LineColumn(1, 1),
                      source='one.js')
    theMap.addMapping(generated=LineColumn(1, 5),
                      original=LineColumn(1, 5),
                      source='one.js')
    theMap.addMapping(generated=LineColumn(1, 9),
                      original=LineColumn(1, 11),
                      source='one.js')
    theMap.addMapping(generated=LineColumn(1, 18),
                      original=LineColumn(1, 21),
                      source='one.js',
                      name='bar')
    theMap.addMapping(generated=LineColumn(1, 21),
                      original=LineColumn(2, 3),
                      source='one.js')
    theMap.addMapping(generated=LineColumn(1, 28),
                      original=LineColumn(2, 10),
                      source='one.js',
                      name='baz')
    theMap.addMapping(generated=LineColumn(1, 32),
                      original=LineColumn(2, 14),
                      source='one.js',
                      name='bar')
    theMap.addMapping(generated=LineColumn(2, 1),
                      original=LineColumn(1, 1),
                      source='two.js')
    theMap.addMapping(generated=LineColumn(2, 5),
                      original=LineColumn(1, 5),
                      source='two.js')
    theMap.addMapping(generated=LineColumn(2, 9),
                      original=LineColumn(1, 11),
                      source='two.js')
    theMap.addMapping(generated=LineColumn(2, 18),
                      original=LineColumn(1, 21),
                      source='two.js',
                      name='n')
    theMap.addMapping(generated=LineColumn(2, 21),
                      original=LineColumn(2, 3),
                      source='two.js')
    theMap.addMapping(generated=LineColumn(2, 28),
                      original=LineColumn(2, 10),
                      source='two.js',
                      name='n')

    mapData = theMap.toJson()
    self.assertEqualMaps(mapData, SampleData.mapData)

  #-----------------------------------------------------------------
  def testThatSourceContentCanBeSet(self):

    theMap = SourceMapGenerator(file='min.js', sourceRoot='/the/root')

    theMap.addMapping(generated=LineColumn(1, 1),
                      original=LineColumn(1, 1),
                      source='one.js')
    theMap.addMapping(generated=LineColumn(2, 1),
                      original=LineColumn(1, 1),
                      source='two.js')

    theMap.setSourceContent('one.js', 'one file content')

    mapData = theMap.toJson()

    self.assertEqual(mapData['sources'][0], 'one.js')
    self.assertEqual(mapData['sources'][1], 'two.js')
    self.assertEqual(mapData['sourcesContent'][0], 'one file content')
    self.assertEqual(mapData['sourcesContent'][1], None)

  #-----------------------------------------------------------------
  def testFromSourceMap(self):

    consumer = SourceMapConsumer(SampleData.mapData)
    theMap = SourceMapGenerator.fromSourceMap(consumer)

    self.assertEqualMaps(theMap.toJson(), SampleData.mapData)

  #-----------------------------------------------------------------
  def testFromSourceMapWithSourcesContent(self):

    consumer = SourceMapConsumer(SampleData.mapWithSourcesContent)
    theMap = SourceMapGenerator.fromSourceMap(consumer)
    self.assertEqualMaps(theMap.toJson(), SampleData.mapWithSourcesContent)

  #-----------------------------------------------------------------
  def testApplySourceMap(self):

    node = SourceNode(None, None, None, [
      SourceNode(2, 0, 'fileX', 'lineX2\n'),
      'genA1\n',
      SourceNode(2, 0, 'fileY', 'lineY2\n'),
      'genA2\n',
      SourceNode(1, 0, 'fileX', 'lineX1\n'),
      'genA3\n',
      SourceNode(1, 0, 'fileY', 'lineY1\n')
    ])
    dummyCode, mapStep1 = node.toStringWithSourceMap(file='fileA')
    mapStep1.setSourceContent('fileX', 'lineX1\nlineX2\n')
    mapStep1 = mapStep1.toJson()

    node = SourceNode(None, None, None, [
      'gen1\n',
      SourceNode(1, 0, 'fileA', 'lineA1\n'),
      SourceNode(2, 0, 'fileA', 'lineA2\n'),
      SourceNode(3, 0, 'fileA', 'lineA3\n'),
      SourceNode(4, 0, 'fileA', 'lineA4\n'),
      SourceNode(1, 0, 'fileB', 'lineB1\n'),
      SourceNode(2, 0, 'fileB', 'lineB2\n'),
      'gen2\n'
    ])
    dummyCode, mapStep2 = node.toStringWithSourceMap(file='fileGen')
    mapStep2.setSourceContent('fileB', 'lineB1\nlineB2\n')
    mapStep2 = mapStep2.toJson()

    node = SourceNode(None, None, None, [
      'gen1\n',
      SourceNode(2, 0, 'fileX', 'lineA1\n'),
      SourceNode(2, 0, 'fileA', 'lineA2\n'),
      SourceNode(2, 0, 'fileY', 'lineA3\n'),
      SourceNode(4, 0, 'fileA', 'lineA4\n'),
      SourceNode(1, 0, 'fileB', 'lineB1\n'),
      SourceNode(2, 0, 'fileB', 'lineB2\n'),
      'gen2\n'
    ])
    dummyCode, expectedMap = node.toStringWithSourceMap(file='fileGen')
    expectedMap.setSourceContent('fileX', 'lineX1\nlineX2\n')
    expectedMap.setSourceContent('fileB', 'lineB1\nlineB2\n')
    expectedMap = expectedMap.toJson()

    # apply source map "mapStep1" to "mapStep2"
    generator = SourceMapGenerator.fromSourceMap(SourceMapConsumer(mapStep2))
    generator.applySourceMap(SourceMapConsumer(mapStep1))
    actualMap = generator.toJson()

    self.assertEqualMaps(actualMap, expectedMap)

  #-----------------------------------------------------------------
  def testSortingWithDuplicateGeneratedMappings(self):

    theMap = SourceMapGenerator(file='test.js')

    theMap.addMapping(generated=LineColumn(3, 0),
                      original=LineColumn(2, 0),
                      source='a.js')
    theMap.addMapping(generated=LineColumn(2, 0))
    theMap.addMapping(generated=LineColumn(2, 0))
    theMap.addMapping(generated=LineColumn(1, 0),
                      original=LineColumn(1, 0),
                      source='a.js')

    mapData = theMap.toJson()

    expected = {
      'version': 3,
      'file': 'test.js',
      'sources': ['a.js'],
      'names': [],
      'mappings': 'AAAA;A;AACA'
    }

    self.assertEqualMaps(mapData, expected)

  #-----------------------------------------------------------------
  def testIgnoreDuplicateMappings(self):

    map1 = SourceMapGenerator(file='min.js', sourceRoot='/the/root')
    map2 = SourceMapGenerator(file='min.js', sourceRoot='/the/root')

    # null original source location

    nullMapping1 = LineColumn(1, 0)
    nullMapping2 = LineColumn(2, 2)

    map1.addMapping(generated=nullMapping1)
    map1.addMapping(generated=nullMapping1)

    map2.addMapping(generated=nullMapping1)

    self.assertEqualMaps(map1.toJson(), map2.toJson())

    map1.addMapping(generated=nullMapping2)
    map1.addMapping(generated=nullMapping2)

    map2.addMapping(generated=nullMapping2)

    self.assertEqualMaps(map1.toJson(), map2.toJson())

    map1 = SourceMapGenerator(file='min.js', sourceRoot='/the/root')
    map2 = SourceMapGenerator(file='min.js', sourceRoot='/the/root')


    # original source location
    srcMapping1 = {
      'generated': LineColumn(1, 0),
      'original': LineColumn(11, 0),
      'source': 'srcMapping1.js'
    }
    srcMapping2 = {
      'generated': LineColumn(2, 2),
      'original': LineColumn(11, 0),
      'source': 'srcMapping2.js'
    }

    map1.addMapping(**srcMapping1)
    map1.addMapping(**srcMapping1)

    map2.addMapping(**srcMapping1)

    self.assertEqualMaps(map1.toJson(), map2.toJson())

    map1.addMapping(**srcMapping2)
    map1.addMapping(**srcMapping1)

    map2.addMapping(**srcMapping2)

    self.assertEqualMaps(map1.toJson(), map2.toJson())

    # full original source and name information
    fullMapping1 = {
      'generated': LineColumn(1, 0),
      'original': LineColumn(11, 0),
      'source': 'fullMapping1.js',
      'name': 'fullMapping1'
    }
    fullMapping2 = {
      'generated': LineColumn(2, 2),
      'original': LineColumn(11, 0),
      'source': 'fullMapping2.js',
      'name': 'fullMapping2'
    }

    map1 = SourceMapGenerator(file='min.js', sourceRoot='/the/root')
    map2 = SourceMapGenerator(file='min.js', sourceRoot='/the/root')

    map1.addMapping(**fullMapping1)
    map1.addMapping(**fullMapping1)

    map2.addMapping(**fullMapping1)

    self.assertEqualMaps(map1.toJson(), map2.toJson())

    map1.addMapping(**fullMapping2)
    map1.addMapping(**fullMapping1)

    map2.addMapping(**fullMapping2)

    self.assertEqualMaps(map1.toJson(), map2.toJson())


  #-----------------------------------------------------------------
  def testCheckForDuplicateNamesOrSources(self):


    theMap = SourceMapGenerator(file='test.js')

    theMap.addMapping(generated=LineColumn(1, 1),
                      original=LineColumn(2, 2),
                      source='a.js',
                      name='foo')
    theMap.addMapping(generated=LineColumn(3, 3),
                      original=LineColumn(4, 4),
                      source='a.js',
                      name='foo')

    expected = {
      'version': 3,
      'file': 'test.js',
      'sources': ['a.js'],
      'names': ['foo'],
      'mappings': 'CACEA;;GAEEA'
    }

    self.assertEqualMaps(theMap.toJson(), expected)

  #-----------------------------------------------------------------
  assertEqualMaps = SampleData.assertEqualMaps
