#-------------------------------------------------------------------
#  TestSourceMapConsumer.py
#
#  The TestSourceMapConsumer module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the SourceMapConsumer class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.js.jsSourceMap import LineColumn
from ai.axe.js.jsSourceMap import SourceLineColumn
from ai.axe.js.jsSourceMap import SourceMapConsumer
from ai.axe.js.jsSourceMap import SourceMapException
from ai.axe.js.jsSourceMap import SourceMapGenerator
from ai.axe.js.jsSourceMap.test import SampleData
from ai.axe.build.unittest import AxeSimpleTestCase
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestSourceMapConsumer(AxeSimpleTestCase):
  '''Unit test for the SourceMapConsumer class.
'''

  #-----------------------------------------------------------------
  def setUp(self):

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self):

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testThatWeCanInstantiateWithAMap(self):

    SourceMapConsumer(SampleData.mapData)

  #-----------------------------------------------------------------
  def testSourcesFieldHasOriginalSources(self):

    theMap = SourceMapConsumer(SampleData.mapData)
    sources = theMap.sources

    self.assertEqual(sources[0], '/the/root/one.js')
    self.assertEqual(sources[1], '/the/root/two.js')
    self.assertEqual(len(sources), 2)

  #-----------------------------------------------------------------
  def testSourceRootIsReflectedInMappingsSourceField(self):

    theMap = SourceMapConsumer(SampleData.mapData)

    mapping = theMap.originalPositionFor(LineColumn(2, 1))
    self.assertEqual(mapping.source, '/the/root/two.js')

    mapping = theMap.originalPositionFor(LineColumn(1, 1))
    self.assertEqual(mapping.source, '/the/root/one.js')

  #-----------------------------------------------------------------
  def testMappingTokensBackExactly(self):

    theMap = SourceMapConsumer(SampleData.mapData)

    self.assertMapping(1, 1, '/the/root/one.js', 1, 1, None, theMap)
    self.assertMapping(1, 5, '/the/root/one.js', 1, 5, None, theMap)
    self.assertMapping(1, 9, '/the/root/one.js', 1, 11, None, theMap)
    self.assertMapping(1, 18, '/the/root/one.js', 1, 21, 'bar', theMap)
    self.assertMapping(1, 21, '/the/root/one.js', 2, 3, None, theMap)
    self.assertMapping(1, 28, '/the/root/one.js', 2, 10, 'baz', theMap)
    self.assertMapping(1, 32, '/the/root/one.js', 2, 14, 'bar', theMap)

    self.assertMapping(2, 1, '/the/root/two.js', 1, 1, None, theMap)
    self.assertMapping(2, 5, '/the/root/two.js', 1, 5, None, theMap)
    self.assertMapping(2, 9, '/the/root/two.js', 1, 11, None, theMap)
    self.assertMapping(2, 18, '/the/root/two.js', 1, 21, 'n', theMap)
    self.assertMapping(2, 21, '/the/root/two.js', 2, 3, None, theMap)
    self.assertMapping(2, 28, '/the/root/two.js', 2, 10, 'n', theMap)

  #-----------------------------------------------------------------
  def testMappingTokensBackFuzzy(self):

    theMap = SourceMapConsumer(SampleData.mapData)

    # Finding original positions
    self.assertMapping(1, 20, '/the/root/one.js', 1, 21, 'bar', theMap, True)
    self.assertMapping(1, 30, '/the/root/one.js', 2, 10, 'baz', theMap, True)
    self.assertMapping(2, 12, '/the/root/two.js', 1, 11, None, theMap, True)

    # Finding generated positions
    self.assertMapping(1, 18, '/the/root/one.js', 1, 22, 'bar', theMap, False,
                       True)
    self.assertMapping(1, 28, '/the/root/one.js', 2, 13, 'baz', theMap, False,
                       True)
    self.assertMapping(2, 9, '/the/root/two.js', 1, 16, None, theMap, False,
                       True)

  #-----------------------------------------------------------------
  def testEachMapping(self):

    theMap = SourceMapConsumer(SampleData.mapData)

    class Callback:

      def __init__(self, test):

        self.previousLine = -1
        self.previousColumn = -1
        self.test = test

        # Keep a running total of line numbers, to make sure the callback
        # is being called for every mapping.
        self.totalLineNumbers = 0

      def __call__(self, mapping):

        self.totalLineNumbers += mapping.generatedLine

        self.test.assertTrue(mapping.generatedLine >= self.previousLine)

        if mapping.source:
          expectedSourceRoot = SampleData.mapData['sourceRoot']
          self.test.assertTrue(mapping.source.startswith(expectedSourceRoot))

        if mapping.generatedLine == self.previousLine:
          self.test.assertTrue(mapping.generatedColumn >= self.previousColumn)
          self.previousColumn = mapping.generatedColumn

        else:
          self.previousLine = mapping.generatedLine
          self.previousColumn = -1

    callback = Callback(self)
    theMap.eachMapping(callback)

    self.assertEqual(19, callback.totalLineNumbers, 'totalLineNumbers')

  #-----------------------------------------------------------------
  def testIteratingOverMappingsInDifferentOrder(self):

    theMap = SourceMapConsumer(SampleData.mapData)

    class Callback:

      def __init__(self, test):

        self.previousLine = -1
        self.previousColumn = -1
        self.previousSource = ""
        self.test = test

        # Keep a running total of line numbers, to make sure the callback
        # is being called for every mapping.
        self.totalLineNumbers = 0

      def __call__(self, mapping):

        self.totalLineNumbers += mapping.generatedLine

        self.test.assertTrue(mapping.source >= self.previousSource)

        if mapping.source == self.previousSource:
          self.test.assertTrue(mapping.originalLine >= self.previousLine)

          if mapping.originalLine == self.previousLine:
            self.test.assertTrue(mapping.originalColumn >= self.previousColumn)
            self.previousColumn = mapping.originalColumn

          else:
            self.previousLine = mapping.originalLine
            self.previousColumn = -1

        else:
          self.previousSource = mapping.source
          self.previousLine = -1
          self.previousColumn = -1


    callback = Callback(self)
    theMap.eachMapping(callback, SourceMapConsumer.ORIGINAL_ORDER)

    self.assertEqual(19, callback.totalLineNumbers, 'totalLineNumbers')

  #-----------------------------------------------------------------
  def testSourcesContentFieldHasTheOriginalSources(self):

    theMap = SourceMapConsumer(SampleData.mapWithSourcesContent)
    sourcesContent = theMap.sourcesContent

    expected = [
      ' ONE.foo = function (bar) {\n   return baz(bar);\n };',
    ' TWO.inc = function (n) {\n   return n + 1;\n };'
    ]
    self.assertEqual(sourcesContent, expected)

  #-----------------------------------------------------------------
  def testThatWeCanGetTheOriginalSourcesForTheSources(self):

    theMap = SourceMapConsumer(SampleData.mapWithSourcesContent)

    sources = theMap.sources

    self.assertEqual(theMap.sourceContentFor(sources[0]),
                     ' ONE.foo = function (bar) {\n   return baz(bar);\n };')
    self.assertEqual(theMap.sourceContentFor(sources[1]),
                     ' TWO.inc = function (n) {\n   return n + 1;\n };')
    self.assertEqual(theMap.sourceContentFor("one.js"),
                     ' ONE.foo = function (bar) {\n   return baz(bar);\n };')
    self.assertEqual(theMap.sourceContentFor("two.js"),
                     ' TWO.inc = function (n) {\n   return n + 1;\n };')

    try:
      theMap.sourceContentFor("")
      self.fail("No exception for empty string.")
    except SourceMapException:
      pass

    try:
      theMap.sourceContentFor("/the/root/three.js")
      self.fail("No exception for bad file absolute path.")
    except SourceMapException:
      pass

    try:
      theMap.sourceContentFor("three.js")
      self.fail("No exception for bad file name.")
    except SourceMapException:
      pass

  #-----------------------------------------------------------------
  def testSourceRootGeneratedPositionFor(self):

    theMap = SourceMapGenerator(file='baz.js', sourceRoot='foo/bar')

    theMap.addMapping(original=LineColumn(1, 1),
                      generated=LineColumn(2, 2),
                      source='bang.coffee')
    theMap.addMapping(original=LineColumn(5, 5),
                      generated=LineColumn(6, 6),
                      source='bang.coffee')

    theMap = SourceMapConsumer(theMap.toJson())

    # Should handle without sourceRoot.
    pos = theMap.generatedPositionFor(SourceLineColumn('bang.coffee', 1, 1))
    self.assertEqual(pos.line, 2)
    self.assertEqual(pos.column, 2)

    # Should handle with sourceRoot.
    pos = theMap.generatedPositionFor(SourceLineColumn('foo/bar/bang.coffee',
                                                       1, 1))
    self.assertEqual(pos.line, 2)
    self.assertEqual(pos.column, 2)

  #-----------------------------------------------------------------
  def testSourceRootOriginalPostitionFor(self):

    theMap = SourceMapGenerator(file='baz.js', sourceRoot='foo/bar')

    theMap.addMapping(original=LineColumn(1, 1),
                      generated=LineColumn(2, 2),
                      source='bang.coffee')

    theMap = SourceMapConsumer(theMap.toJson())

    pos = theMap.originalPositionFor(LineColumn(2, 2))

    # Should always have the prepended source root
    self.assertEqual(pos.source, 'foo/bar/bang.coffee')
    self.assertEqual(pos.line, 1)
    self.assertEqual(pos.column, 1)

  #-----------------------------------------------------------------
  def testGithubIssue56(self):

    theMap = SourceMapGenerator(file='www.example.com/foo.js',
                                sourceRoot='http://')

    theMap.addMapping(original=LineColumn(1, 1),
                      generated=LineColumn(2, 2),
                      source='www.example.com/original.js')

    theMap = SourceMapConsumer(theMap.toJson())


    sources = theMap.sources
    self.assertEqual(sources, ['http://www.example.com/original.js'])

  # MPB commented out because not using URLs right now.
#  #-----------------------------------------------------------------
#  def testGithubIssue43(self):
#
#    theMap = SourceMapGenerator(sourceRoot='http://www.example.com',
#                                file='foo.js')
#
#    theMap.addMapping(original=LineColumn(1, 1),
#                      generated=LineColumn(2, 2),
#                      source = 'http://cdn.example.com/original.js')
#
#    theMap = SourceMapConsumer(theMap.toJson())
#
#    sources = theMap.sources
#    self.assertEqual(len(sources), 1,
#                     'Should only be one source.')
#    self.assertEqual(sources[0], 'http://cdn.example.com/original.js',
#                     'Should not be joined with the sourceRoot.')

  # MPB commented out because not using URLs right now.
#  #-----------------------------------------------------------------
#  def testAbsolutePathButSomeHostSources(self):
#
#    theMap = SourceMapGenerator(file='foo.js',
#                                sourceRoot='http://example.com/foo/bar')
#
#    theMap.addMapping(original=LineColumn(1, 1),
#                      generated=LineColumn(2, 2),
#                      source = '/original.js')
#
#
#    theMap = SourceMapConsumer(theMap.toJson())
#
#    sources = theMap.sources
#    self.assertEqual(len(sources), 1,
#                     'Should only be one source.')
#    self.assertEqual(sources[0], 'http://example.com/original.js',
#                     'Source should be relative the host of the source root.')


#  exports['test github issue #64'] = function (assert, util) {
#    var map = new SourceMapConsumer({
#      "version": 3,
#      "file": "foo.js",
#      "sourceRoot": "http://example.com/",
#      "sources": ["/a"],
#      "names": [],
#      "mappings": "AACA",
#      "sourcesContent": ["foo"]
#    })
#
#    self.assertEqual(map.sourceContentFor("a"), "foo")
#    self.assertEqual(map.sourceContentFor("/a"), "foo")
#  }
#
#  exports['test bug 885597'] = function (assert, util) {
#    var map = new SourceMapConsumer({
#      "version": 3,
#      "file": "foo.js",
#      "sourceRoot": "file:///Users/AlGore/Invented/The/Internet/",
#      "sources": ["/a"],
#      "names": [],
#      "mappings": "AACA",
#      "sourcesContent": ["foo"]
#    })
#
#    var s = map.sources[0]
#    self.assertEqual(map.sourceContentFor(s), "foo")
#  }
#
#  exports['test github issue #72, duplicate sources'] =
#    function (assert, util) {
#    var map = new SourceMapConsumer({
#      "version": 3,
#      "file": "foo.js",
#      "sources": ["source1.js", "source1.js", "source3.js"],
#      "names": [],
#      "mappings": ";EAAC;;IAEE;;MEEE",
#      "sourceRoot": "http://example.com"
#    })
#
#    var pos = map.originalPositionFor({
#      line: 2,
#      column: 2
#    })
#    self.assertEqual(pos.source, 'http://example.com/source1.js')
#    self.assertEqual(pos.line, 1)
#    self.assertEqual(pos.column, 1)
#
#    var pos = map.originalPositionFor({
#      line: 4,
#      column: 4
#    })
#    self.assertEqual(pos.source, 'http://example.com/source1.js')
#    self.assertEqual(pos.line, 3)
#    self.assertEqual(pos.column, 3)
#
#    var pos = map.originalPositionFor({
#      line: 6,
#      column: 6
#    })
#    self.assertEqual(pos.source, 'http://example.com/source3.js')
#    self.assertEqual(pos.line, 5)
#    self.assertEqual(pos.column, 5)
#  }
#
#  exports['test github issue #72, duplicate names'] = function (assert, util) {
#    var map = new SourceMapConsumer({
#      "version": 3,
#      "file": "foo.js",
#      "sources": ["source.js"],
#      "names": ["name1", "name1", "name3"],
#      "mappings": ";EAACA;;IAEEA;;MAEEE",
#      "sourceRoot": "http://example.com"
#    })
#
#    var pos = map.originalPositionFor({
#      line: 2,
#      column: 2
#    })
#    self.assertEqual(pos.name, 'name1')
#    self.assertEqual(pos.line, 1)
#    self.assertEqual(pos.column, 1)
#
#    var pos = map.originalPositionFor({
#      line: 4,
#      column: 4
#    })
#    self.assertEqual(pos.name, 'name1')
#    self.assertEqual(pos.line, 3)
#    self.assertEqual(pos.column, 3)
#
#    var pos = map.originalPositionFor({
#      line: 6,
#      column: 6
#    })
#    self.assertEqual(pos.name, 'name3')
#    self.assertEqual(pos.line, 5)
#    self.assertEqual(pos.column, 5)
#  }

  #-----------------------------------------------------------------
  def testSourceMapConsumerFromSourceMap(self):


    smg = SourceMapGenerator(file='foo.js',
                             sourceRoot='http://example.com/')

    smg.addMapping(original=LineColumn(1, 1),
                   generated=LineColumn(2, 2),
                   source='bar.js')

    smg.addMapping(original=LineColumn(2, 2),
                   generated=LineColumn(4, 4),
                   source='baz.js',
                   name='dirtMcGirt')

    smg.setSourceContent('baz.js', 'baz.js content')

    smc = SourceMapConsumer.fromSourceMap(smg)

    self.assertEqual(smc.file, 'foo.js')
    self.assertEqual(smc.sourceRoot, 'http://example.com/')
    self.assertEqual(len(smc.sources), 2)
    self.assertEqual(smc.sources[0], 'http://example.com/bar.js')
    self.assertEqual(smc.sources[1], 'http://example.com/baz.js')
    self.assertEqual(smc.sourceContentFor('baz.js'), 'baz.js content')

    pos = smc.originalPositionFor(LineColumn(2, 2))
    self.assertEqual(pos.line, 1)
    self.assertEqual(pos.column, 1)
    self.assertEqual(pos.source, 'http://example.com/bar.js')
    self.assertEqual(pos.name, None)

    pos = smc.generatedPositionFor(SourceLineColumn('http://example.com/bar.js',
                                                    1, 1))
    self.assertEqual(pos.line, 2)
    self.assertEqual(pos.column, 2)

    pos = smc.originalPositionFor(LineColumn(4, 4))
    self.assertEqual(pos.line, 2)
    self.assertEqual(pos.column, 2)
    self.assertEqual(pos.source, 'http://example.com/baz.js')
    self.assertEqual(pos.name, 'dirtMcGirt')

    pos = smc.generatedPositionFor(SourceLineColumn('http://example.com/baz.js',
                                                    2, 2))
    self.assertEqual(pos.line, 4)
    self.assertEqual(pos.column, 4)

  #-----------------------------------------------------------------
  assertMapping = SampleData.assertMapping
