#-------------------------------------------------------------------
#  VlqBase64.py
#
#  The VlqBase64 module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Variable length quantity encoding/decoding.

A single base 64 digit can contain 6 bits of data. For the base 64 variable
length quantities we use in the source map spec, the first bit is the sign,
the next four bits are the actual value, and the 6th bit is the
continuation bit. The continuation bit tells us whether there are more
digits in this value following this digit.

  Continuation
  |    Sign
  |    |
  V    V
  101011
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import Base64
from .VlqValue import VlqValue
from .SourceMapException import SourceMapException
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
VLQ_BASE_SHIFT = 5

#-------------------------------------------------------------------
# binary: 100000
VLQ_BASE = 1 << VLQ_BASE_SHIFT

#-------------------------------------------------------------------
# binary: 011111
VLQ_BASE_MASK = VLQ_BASE - 1

#-------------------------------------------------------------------
# binary: 100000
VLQ_CONTINUATION_BIT = VLQ_BASE

#-------------------------------------------------------------------
def toVLQSigned(aValue):
  '''Converts from a two-complement value to a value where the sign bit is
  is placed in the least significant bit.  For example, as decimals:
  1 becomes 2 (10 binary), -1 becomes 3 (11 binary)
  2 becomes 4 (100 binary), -2 becomes 5 (101 binary)
  '''

  if aValue < 0:
    return ((-aValue) << 1) + 1
  else:
    return (aValue << 1) + 0

#-------------------------------------------------------------------
def fromVLQSigned(aValue):
  ''' Converts to a two-complement value from a value where the sign bit is
  is placed in the least significant bit.  For example, as decimals:
  2 (10 binary) becomes 1, 3 (11 binary) becomes -1
  4 (100 binary) becomes 2, 5 (101 binary) becomes -2
  '''
  isNegative = (aValue & 1) == 1
  shifted = aValue >> 1

  if isNegative:
    return -shifted
  else:
    return shifted

#-------------------------------------------------------------------
def encode(aValue):
  ''' Returns the base 64 VLQ encoded value.
  '''

  encoded = ""

  vlq = toVLQSigned(aValue)

  while True:
    digit = vlq & VLQ_BASE_MASK
    vlq = vlq >> VLQ_BASE_SHIFT
    if vlq > 0:
      # There are still more digits in this value, so we must make sure the
      # continuation bit is marked.
      digit |= VLQ_CONTINUATION_BIT
    encoded += Base64.encode(digit)
    if not vlq > 0:
      break

  return encoded

#-------------------------------------------------------------------
def decode(aStr):
  '''Decodes the next base 64 VLQ value from the given string and returns the
  value and the rest of the string.
  '''

  i = 0
  strLen = len(aStr)
  result = 0
  shift = 0

  while True:
    if i >= strLen:
      raise SourceMapException("Expected more digits in base 64 VLQ value.")

    digit = Base64.decode(aStr[i])
    i += 1
    continuation = bool(digit & VLQ_CONTINUATION_BIT)
    digit &= VLQ_BASE_MASK
    result = result + (digit << shift)
    shift += VLQ_BASE_SHIFT
    if not continuation:
      break

  value = fromVLQSigned(result)
  rest = aStr[i:]

  return VlqValue(value, rest)
