#-------------------------------------------------------------------
#  SourceNode.py
#
#  The SourceNode class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the SourceNode class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import SourceMapGenerator
from . import JsUtil
from .LineColumn import LineColumn
from .SourceLineColumn import SourceLineColumn
from .SourceMapException import SourceMapException
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class SourceNode:
  '''SourceNodes provide a way to abstract over interpolating/concatenating
  snippets of generated JavaScript source code while maintaining the line and
  column information associated with the original source code.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               aLine=None,
               aColumn=None,
               aSource=None,
               aChunks=None,
               aName=None):
    '''Creates a new SourceNode.

    @param aLine The original line number.
    @param aColumn The original column number.
    @param aSource The original source's filename.
    @param aChunks Optional. An array of strings which are snippets of
           generated JS, or other SourceNodes.
    @param aName The original identifier.
    '''

    self.children = []
    self.sourceContents = {}
    self.line = aLine
    self.column = aColumn
    self.source = aSource
    self.name = aName

    if aChunks:
      self.add(aChunks)

  #-----------------------------------------------------------------
  @staticmethod
  def fromStringWithSourceMap(aGeneratedCode, aSourceMapConsumer):
    '''Creates a SourceNode from generated code and a SourceMapConsumer.

    @param aGeneratedCode The generated code
    @param aSourceMapConsumer The SourceMap for the generated code
    '''

    # The SourceNode we want to fill with the generated code
    # and the SourceMap
    node = SourceNode()
    nodeAdder = NodeAdder(node)

    # The generated code
    # Processed fragments are removed from this array.
    remainingLines = aGeneratedCode.split('\n')

    # We need to remember the position of "remainingLines"
    lastGeneratedLine = 1
    lastGeneratedColumn = 0

    # The generate SourceNodes we need a code range.
    # To extract it current and last mapping is used.
    # Here we store the last mapping.
    lastMapping = None

    for mapping in aSourceMapConsumer.addRootToMappings():
      if lastMapping is None:
        # We add the generated code until the first mapping
        # to the SourceNode without any mapping.
        # Each line is added as separate string.
        while lastGeneratedLine < mapping.generatedLine:
          node.add(remainingLines.pop(0) + "\n")
          lastGeneratedLine += 1

        if lastGeneratedColumn < mapping.generatedColumn:
          nextLine = remainingLines[0]
          node.add(nextLine[0:mapping.generatedColumn])
          remainingLines[0] = nextLine[mapping.generatedColumn:]
          lastGeneratedColumn = mapping.generatedColumn

      else:
        # We add the code from "lastMapping" to "mapping":
        # First check if there is a new line in between.
        if lastGeneratedLine < mapping.generatedLine:
          code = ""
          # Associate full lines with "lastMapping"
          while True:
            code += remainingLines.pop(0) + "\n"
            lastGeneratedLine += 1
            lastGeneratedColumn = 0
            if not lastGeneratedLine < mapping.generatedLine:
              break

          # When we reached the correct line, we add code until we
          # reach the correct column too.
          if lastGeneratedColumn < mapping.generatedColumn:
            nextLine = remainingLines[0]
            code += nextLine[0:mapping.generatedColumn]
            remainingLines[0] = nextLine[mapping.generatedColumn:]
            lastGeneratedColumn = mapping.generatedColumn

          # Create the SourceNode.
          nodeAdder.addMappingWithCode(lastMapping, code)
        else:
          # There is no new line in between.
          # Associate the code between "lastGeneratedColumn" and
          # "mapping.generatedColumn" with "lastMapping"
          nextLine = remainingLines[0]
          code = nextLine[0:mapping.generatedColumn - lastGeneratedColumn]

          index = mapping.generatedColumn - lastGeneratedColumn
          remainingLines[0] = nextLine[index:]

          lastGeneratedColumn = mapping.generatedColumn
          nodeAdder.addMappingWithCode(lastMapping, code)

      lastMapping = mapping

    # We have processed all mappings.
    # Associate the remaining code in the current line with "lastMapping"
    # and add the remaining lines without any mapping
    nodeAdder.addMappingWithCode(lastMapping, "\n".join(remainingLines))

    # Copy sourcesContent into SourceNode
    for sourceFile in aSourceMapConsumer.sources:
      content = aSourceMapConsumer.sourceContentFor(sourceFile)
      if content:
        node.setSourceContent(sourceFile, content)

    return node

  #-----------------------------------------------------------------
  def add(self, aChunk):
    '''Add a chunk of generated JS to this source node.

    @param aChunk A string snippet of generated JS code, another instance of
           SourceNode, or an array where each member is one of those things.
    '''

    if isinstance(aChunk, (list, tuple)):
      for chunk in aChunk:
        self.add(chunk)

    elif isinstance(aChunk, (SourceNode, str)):
      if aChunk:
        self.children.append(aChunk)

    else:
      raise SourceMapException(
        "Expected a SourceNode, string, or an array of " +
        "SourceNodes and strings. Got " + str(aChunk)
      )

    return self

  #-----------------------------------------------------------------
  def prepend(self, aChunk):
    '''Add a chunk of generated JS to the beginning of this source node.

    @param aChunk A string snippet of generated JS code, another instance of
           SourceNode, or an array where each member is one of those things.
    '''

    if isinstance(aChunk, (list, tuple)):
      for chunk in reversed(aChunk):
        self.prepend(chunk)

    elif isinstance(aChunk, (SourceNode, str)):
      self.children.insert(0, aChunk)

    else:
      raise SourceMapException(
        "Expected a SourceNode, string, or an array of " +
        "SourceNodes and strings. Got " + str(aChunk)
      )

    return self

  #-----------------------------------------------------------------
  def walk(self, aFn):
    '''Walk over the tree of JS snippets in this node and its children. The
    walking function is called once for each snippet of JS and is passed that
    snippet and the its original associated source's line/column location.

    @param aFn The traversal function.
    '''

    for chunk in self.children:
      if isinstance(chunk, SourceNode):
        chunk.walk(aFn)

      else:
        if chunk != '':
          aFn(chunk, SourceLineColumn(source=self.source,
                                      line=self.line,
                                      column=self.column,
                                      name=self.name))

  #-----------------------------------------------------------------
  def join(self, aSep):
    '''Like `String.prototype.join` except for SourceNodes.

    Inserts `aStr` between each of `self.children`.

    @param aSep The separator.
    '''

    if self.children:

      newChildren = []

      for child in self.children[:-1]:
        newChildren.append(child)
        newChildren.append(aSep)
      newChildren.append(self.children[-1])

      self.children = newChildren

    return self

  #-----------------------------------------------------------------
  def replaceRight(self, aPattern, aReplacement):
    '''Call String.prototype.replace on the very right-most source snippet.

    Useful
    for trimming whitespace from the end of a source node, etc.

    @param aPattern The pattern to replace.
    @param aReplacement The thing to replace the pattern with.
    '''

    lastChild = self.children[-1]
    if isinstance(lastChild, SourceNode):
      lastChild.replaceRight(aPattern, aReplacement)

    elif isinstance(lastChild, str):
      self.children[-1] = lastChild.replace(aPattern, aReplacement)

    else:
      self.children.append(''.replace(aPattern, aReplacement))

    return self

  #-----------------------------------------------------------------
  def setSourceContent(self, aSourceFile, aSourceContent):
    '''Set the source content for a source file.

    This will be added to the SourceMapGenerator
    in the sourcesContent field.

    @param aSourceFile The filename of the source file
    @param aSourceContent The content of the source file
    '''

    self.sourceContents[JsUtil.toSetString(aSourceFile)] = aSourceContent

  #-----------------------------------------------------------------
  def walkSourceContents(self, aFn):
    '''Walk over the tree of SourceNodes.

    The walking function is called for each
    source file content and is passed the filename and source content.

    @param aFn The traversal function.
    '''

    for child in self.children:
      if isinstance(child, SourceNode):
        child.walkSourceContents(aFn)

    sources = list(self.sourceContents.keys())
    for sourceItem in sources:
      aFn(JsUtil.fromSetString(sourceItem), self.sourceContents[sourceItem])

  #-----------------------------------------------------------------
  def toString(self):
    '''Return the string representation of this source node. Walks over the tree
    and concatenates all the various snippets together to one string.
    '''

    collector = StringCollector()
    self.walk(collector)
    return collector.theString

  #-----------------------------------------------------------------
  def toStringWithSourceMap(self, **aArgs):
    '''Returns the string representation of this source node along with a source
    map.
    '''

    theMap = SourceMapGenerator(**aArgs)

    builder = GeneratorBuilder(theMap)

    self.walk(builder)
    self.walkSourceContents(builder.setSourceContent)

    return builder.generated['code'], theMap

#===================================================================
class NodeAdder:
  '''Adds nodes to a node.
  '''

  #-----------------------------------------------------------------
  def __init__(self, node):
    '''Creates a new NodeAdder.

    @param node The node to add to.
    '''

    self.node = node

  #-----------------------------------------------------------------
  def addMappingWithCode(self, mapping, code):
    if not mapping or  not mapping.source:
      self.node.add(code)
    else:
      self.node.add(SourceNode(mapping.originalLine,
                               mapping.originalColumn,
                               mapping.source,
                               code,
                               mapping.name))

#===================================================================
class StringCollector:
  '''Collects all strings it is called with.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new StringCollector.
    '''

    self.theString = ''

  #-----------------------------------------------------------------
  def __call__(self, chunk, original):
    self.theString += chunk

#===================================================================
class GeneratorBuilder:
  '''Builds up a SourceMapGenerator.
  '''

  #-----------------------------------------------------------------
  def __init__(self, theMap):
    '''Creates a new StringCollector.
    '''

    self.theMap = theMap

    self.generated = {
      'code': "",
      'line': 1,
      'column': 0
    }

    self.sourceMappingActive = False
    self.lastOriginalSource = None
    self.lastOriginalLine = None
    self.lastOriginalColumn = None
    self.lastOriginalName = None

  #-----------------------------------------------------------------
  def setSourceContent(self, sourceFile, sourceContent):
    self.theMap.setSourceContent(sourceFile, sourceContent)

  #-----------------------------------------------------------------
  def __call__(self, chunk, original):

    self.generated['code'] += chunk
    if (original.source is not None and
        original.line is not None and
        original.column is not None):
      if (self.lastOriginalSource != original.source or
          self.lastOriginalLine != original.line or
          self.lastOriginalColumn != original.column or
          self.lastOriginalName != original.name):

        self.theMap.addMapping(source=original.source,
                          original=LineColumn(original.line, original.column),
                          generated=LineColumn(self.generated['line'],
                                               self.generated['column']),
                          name=original.name)

      self.lastOriginalSource = original.source
      self.lastOriginalLine = original.line
      self.lastOriginalColumn = original.column
      self.lastOriginalName = original.name
      self.sourceMappingActive = True

    elif self.sourceMappingActive:
      self.theMap.addMapping(generated=LineColumn(self.generated['line'],
                                             self.generated['column']))
      self.lastOriginalSource = None
      self.sourceMappingActive = False

    for ch in chunk:
      if ch == '\n':
        self.generated['line'] += 1
        self.generated['column'] = 0
      else:
        self.generated['column'] += 1
