#-------------------------------------------------------------------
#  SourceMapping.py
#
#  The SourceMapping class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the SourceMapping class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from functools import cmp_to_key
from typing import Callable, Any
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
def cmp(a, b):
  '''Replacement for cmp() function removed from Python 3.'''

  if a == b:
    return 0
  elif a < b:
    return -1
  else:
    return 1

#===================================================================
class SourceMapping:
  '''A single source file mapping.
  '''

  #-----------------------------------------------------------------
  # Key version of the cmp function, for use in Python 3 list.sort().
  compareByOriginalPositionsKey: Callable[['SourceMapping'], Any]

  #-------------------------------------------------------------------
  # Key version of the cmp function, for use in Python 3 list.sort().
  compareByGeneratedPositionsKey: Callable[['SourceMapping'], Any]

  #-----------------------------------------------------------------
  def __init__(self,
               generatedLine,
               generatedColumn,
               originalLine,
               originalColumn,
               source,
               name):
    '''Creates a new SourceMapping.
    '''

    self.generatedLine = generatedLine
    self.generatedColumn = generatedColumn
    self.originalLine = originalLine
    self.originalColumn = originalColumn
    self.source = source
    self.name = name

  #-----------------------------------------------------------------
  @staticmethod
  def compareByOriginalPositions(mappingA, mappingB, onlyCompareOriginal=False):
    '''Comparator between two mappings where original positions are compared.

    Optionally pass in `true` as `onlyCompareGenerated` to consider two
    mappings with the same original source/line/column, but different generated
    line and column the same. Useful when searching for a mapping with a
    stubbed out mapping.
    '''

    cmpValue = cmp(mappingA.source, mappingB.source)
    if cmpValue:
      return cmpValue

    cmpValue = mappingA.originalLine - mappingB.originalLine
    if cmpValue:
      return cmpValue

    cmpValue = mappingA.originalColumn - mappingB.originalColumn
    if cmpValue or onlyCompareOriginal:
      return cmpValue

    cmpValue = cmp(mappingA.name, mappingB.name)
    if cmpValue:
      return cmpValue

    cmpValue = mappingA.generatedLine - mappingB.generatedLine
    if cmpValue:
      return cmpValue

    return mappingA.generatedColumn - mappingB.generatedColumn

  #-----------------------------------------------------------------
  @staticmethod
  def compareByGeneratedPositions(mappingA,
                                  mappingB,
                                  onlyCompareGenerated=False):
    '''Comparator between two mappings where the generated positions are
    compared.

    Optionally pass in `true` as `onlyCompareGenerated` to consider two
    mappings with the same generated line and column, but different
    source/name/original line and column the same. Useful when searching for a
    mapping with a stubbed out mapping.
    '''

    cmpValue = mappingA.generatedLine - mappingB.generatedLine
    if cmpValue:
      return cmpValue

    cmpValue = mappingA.generatedColumn - mappingB.generatedColumn
    if cmpValue or onlyCompareGenerated:
      return cmpValue

    cmpValue = cmp(mappingA.source, mappingB.source)
    if cmpValue:
      return cmpValue

    if mappingA.originalLine is not None and mappingB.originalLine is not None:
      cmpValue = mappingA.originalLine - mappingB.originalLine
      if cmpValue:
        return cmpValue

    if (mappingA.originalColumn is not None and
        mappingB.originalColumn is not None):
      cmpValue = mappingA.originalColumn - mappingB.originalColumn
      if cmpValue:
        return cmpValue

    return cmp(mappingA.name, mappingB.name)

  #-----------------------------------------------------------------
  def __repr__(self):

    template = 'SourceMapping(generatedLine=%s,generatedColumn=%s,'
    template += 'originalLine=%s,originalColumn=%s,source=%s,name=%s)'

    ret = template % (self.generatedLine,
                      self.generatedColumn,
                      self.originalLine,
                      self.originalColumn,
                      self.source,
                      self.name)

    return ret

#-----------------------------------------------------------------
# Key version of the cmp function, for use in Python 3 list.sort().
SourceMapping.compareByOriginalPositionsKey = (
  cmp_to_key(SourceMapping.compareByOriginalPositions))

#-------------------------------------------------------------------
# Key version of the cmp function, for use in Python 3 list.sort().
SourceMapping.compareByGeneratedPositionsKey = (
  cmp_to_key(SourceMapping.compareByGeneratedPositions))
