#-------------------------------------------------------------------
#  SourceMapGenerator.py
#
#  The SourceMapGenerator class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the SourceMapGenerator class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import LineColumn
from . import VlqBase64
from . import JsUtil
from .SourceMapException import SourceMapException
from .SourceMapping import SourceMapping
import json
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class SourceMapGenerator:
  '''An instance of the SourceMapGenerator represents a source map which is
   being built incrementally. To create a new one, you must pass an object
   with the following properties:

     - file: The filename of the generated source.
     - sourceRoot: An optional root for all URLs in this source map.
  '''

  #-----------------------------------------------------------------
  def __init__(self, file, sourceRoot=None):
    '''Creates a new SourceMapGenerator.
    '''

    self._file = file
    self._sourceRoot = sourceRoot
    self._sources = []
    self._names = []
    self._mappings = []
    self._sourcesContents = None


  #-----------------------------------------------------------------
  @staticmethod
  def fromSourceMap(aSourceMapConsumer):
    '''Creates a new SourceMapGenerator based on a SourceMapConsumer

    @param aSourceMapConsumer The SourceMap.
    '''

    sourceRoot = aSourceMapConsumer.sourceRoot
    generator = SourceMapGenerator(file=aSourceMapConsumer.file,
                                   sourceRoot=sourceRoot)


    for mapping in aSourceMapConsumer.addRootToMappings():

      # Map of args to pass to the addMapping() method.

      newMapping = {}
      newMapping['generated'] = LineColumn(mapping.generatedLine,
                                           mapping.generatedColumn)

      if mapping.source:
        newMapping['source'] = mapping.source
        if sourceRoot:
          newMapping['source'] = JsUtil.relpath(newMapping['source'],
                                                 sourceRoot)

        newMapping['original'] = LineColumn(mapping.originalLine,
                                            mapping.originalColumn)

        if mapping.name:
          newMapping['name'] = mapping.name

      generator.addMapping(**newMapping)

    for sourceFile in aSourceMapConsumer.sources:
      content = aSourceMapConsumer.sourceContentFor(sourceFile)
      if content:
        generator.setSourceContent(sourceFile, content)

    return generator

  #-----------------------------------------------------------------
  def addMapping(self, generated, original=None, source=None, name=None):
    '''Add a single mapping from original source line and column to
    the generated
    source's line and column for this source map being created. The mapping
    object should have the following properties:

     - generated: An object with the generated line and column positions.
     - original: An object with the original line and column positions.
     - source: The original source file (relative to the sourceRoot).
     - name: An optional original token name for this mapping.
    '''

    self._validateMapping(generated, original, source, name)

    if source and not source in self._sources:
      self._sources.append(source)

    if name and not name in self._names:
      self._names.append(name)

    originalLine = None
    originalColumn = None
    if original:
      originalLine = original.line
      originalColumn = original.column

    mapping = SourceMapping(generated.line,
                            generated.column,
                            originalLine,
                            originalColumn,
                            source,
                            name)
    self._mappings.append(mapping)

  #-----------------------------------------------------------------
  def setSourceContent(self, aSourceFile, aSourceContent):
    '''Set the source content for a source file.
    '''

    source = aSourceFile
    if self._sourceRoot:
      source = JsUtil.relpath(source, self._sourceRoot)

    if aSourceContent:
      # Add the source content to the _sourcesContents map.
      # Create a new _sourcesContents map if the property is null.
      if not self._sourcesContents:
        self._sourcesContents = {}
      self._sourcesContents[JsUtil.toSetString(source)] = aSourceContent
    else:
      # Remove the source file from the _sourcesContents map.
      # If the _sourcesContents map is empty, set the property to null.
      del self._sourcesContents[JsUtil.toSetString(source)]
      if not self._sourcesContents:
        self._sourcesContents = None


  #-----------------------------------------------------------------
  # pylint: disable=R0912
  def applySourceMap(self, aSourceMapConsumer, aSourceFile=None):
    '''Applies the mappings of a sub-source-map for a specific source file to
    the
    source map being generated. Each mapping to the supplied source file is
    rewritten using the supplied source map. Note: The resolution for the
    resulting mappings is the minimium of this map and the supplied map.

    @param aSourceMapConsumer The source map to be applied.
    @param aSourceFile Optional. The filename of the source file.
           If omitted, SourceMapConsumer's file property will be used.
    '''

    # If aSourceFile is omitted, we will use the file property of the SourceMap
    if not aSourceFile:
      aSourceFile = aSourceMapConsumer.file

    sourceRoot = self._sourceRoot
    # Make "aSourceFile" relative if an absolute Url is passed.
    if sourceRoot:
      aSourceFile = JsUtil.relpath(aSourceFile, sourceRoot)

    # Applying the SourceMap can add and remove items from the sources and
    # the names array.
    newSources = []
    newNames = []

    # Find mappings for the "aSourceFile"
    for mapping in self._mappings:
      if mapping.source == aSourceFile and mapping.originalLine is not None:
        # Check if it can be mapped by the source map, then update the mapping.
        original = aSourceMapConsumer.originalPositionFor(
          LineColumn(mapping.originalLine, mapping.originalColumn)
        )
        if original.source:
          # Copy mapping
          if sourceRoot:
            mapping.source = JsUtil.relpath(original.source, sourceRoot)
          else:
            mapping.source = original.source

          mapping.originalLine = original.line
          mapping.originalColumn = original.column
          if original.name and mapping.name:
            # Only use the identifier name if it's an identifier
            # in both SourceMaps
            mapping.name = original.name

      source = mapping.source
      if source and not source in newSources:
        newSources.append(source)

      name = mapping.name
      if name and not name in newNames:
        newNames.append(name)

    self._sources = newSources
    self._names = newNames

    # Copy sourcesContents of applied map.
    for sourceFile in aSourceMapConsumer.sources:
      content = aSourceMapConsumer.sourceContentFor(sourceFile)
      if content:
        if sourceRoot:
          sourceFile = JsUtil.relpath(sourceFile, sourceRoot)
        self.setSourceContent(sourceFile, content)

  #-----------------------------------------------------------------
  def _validateMapping(self, aGenerated, aOriginal, aSource, aName):
    '''A mapping can have one of the three levels of data:

     1. Just the generated position.
     2. The Generated position, original position, and original source.
     3. Generated and original position, original source, as well as a name
        token.

   To maintain consistency, we validate that any new mapping being added falls
   in to one of these categories.
   '''

    if (aGenerated and
        aGenerated.line > 0 and aGenerated.column >= 0 and
        not aOriginal and not aSource and not aName):
      # Case 1.
      return

    elif (aGenerated
             and aOriginal
             and aGenerated.line > 0 and aGenerated.column >= 0
             and aOriginal.line > 0 and aOriginal.column >= 0
             and aSource):
      # Cases 2 and 3.
      return

    else:

      msg = ('Invalid mapping: generated=' + str(aGenerated) +
             ' source=' + str(aSource) + ' original=' + str(aOriginal) +
             ' name=' + str(aName))
      raise SourceMapException(msg)

  #-----------------------------------------------------------------
  def _serializeMappings(self):
    '''Serialize the accumulated mappings in to the stream of base 64 VLQs
    specified by the source map format.
    '''

    previousGeneratedColumn = 0
    previousGeneratedLine = 1
    previousOriginalColumn = 0
    previousOriginalLine = 0
    previousName = 0
    previousSource = 0
    result = ''

    # The mappings must be guaranteed to be in sorted order before we start
    # serializing them or else the generated line numbers (which are defined
    # via the ';' separators) will be all messed up. Note: it might be more
    # performant to maintain the sorting as we insert them, rather than as we
    # serialize them, but the big O is the same either way.
    self._mappings.sort(key=SourceMapping.compareByGeneratedPositionsKey)

    for i, mapping in enumerate(self._mappings):

      if mapping.generatedLine != previousGeneratedLine:
        previousGeneratedColumn = 0
        while mapping.generatedLine != previousGeneratedLine:
          result += ';'
          previousGeneratedLine += 1
      else:
        if i > 0:
          if not SourceMapping.compareByGeneratedPositions(mapping,
                                                    self._mappings[i - 1]):
            continue
          result += ','

      result += VlqBase64.encode(mapping.generatedColumn
                                 - previousGeneratedColumn)
      previousGeneratedColumn = mapping.generatedColumn

      if mapping.source:
        result += VlqBase64.encode(self._sources.index(mapping.source)
                                   - previousSource)
        previousSource = self._sources.index(mapping.source)

        # lines are stored 0-based in SourceMap spec version 3
        result += VlqBase64.encode(mapping.originalLine - 1
                                   - previousOriginalLine)
        previousOriginalLine = mapping.originalLine - 1

        result += VlqBase64.encode(mapping.originalColumn
                                   - previousOriginalColumn)
        previousOriginalColumn = mapping.originalColumn

        if mapping.name:
          result += VlqBase64.encode(self._names.index(mapping.name)
                                     - previousName)
          previousName = self._names.index(mapping.name)

    return result

  #-----------------------------------------------------------------
  def _generateSourcesContent(self, aSources, aSourceRoot):

    return [self.getContentsForSource(aSourceRoot, src) for src in aSources]

  #-----------------------------------------------------------------
  def getContentsForSource(self, aSourceRoot, source):

    if not self._sourcesContents:
      return None
    if aSourceRoot:
      source = JsUtil.relpath(source, aSourceRoot)

    key = JsUtil.toSetString(source)

    if key in self._sourcesContents:
      return self._sourcesContents[key]
    else:
      return None

  #-----------------------------------------------------------------
  def toJson(self):
    '''Externalize the source map.
    '''

    theMap = {
      'version': 3,
      'file': self._file,
      'sources': self._sources,
      'names': self._names,
      'mappings': self._serializeMappings()
    }

    if self._sourceRoot:
      theMap['sourceRoot'] = self._sourceRoot

    if self._sourcesContents:
      sourceRoot = self._sourceRoot
      sourcesContent = self._generateSourcesContent(self._sources, sourceRoot)
      theMap['sourcesContent'] = sourcesContent

    return theMap

  #-----------------------------------------------------------------
  def toString(self):
    '''Render the source map being generated to a string.
    '''

    return json.dumps(self.toJson())
