#-------------------------------------------------------------------
#  SourceMapConsumer.py
#
#  The SourceMapConsumer class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the SourceMapConsumer class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import VlqBase64
from . import JsUtil
from .LineColumn import LineColumn
from .SourceLineColumn import SourceLineColumn
from .SourceMapException import SourceMapException
from .SourceMapping import SourceMapping
import os.path
import urllib.parse
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class SourceMapConsumer:
  '''A SourceMapConsumer instance represents a parsed source map which we can
   query for information about the original file positions by giving it a file
   position in the generated source.

   The only parameter is the raw source map (either as a JSON string, or
   already parsed to an object). According to the spec, source maps have the
   following attributes:

     - version: Which version of the source map spec this map is following.
     - sources: An array of URLs to the original source files.
     - names: An array of identifiers which can be referrenced by individual
              mappings.
     - sourceRoot: Optional. The URL root from which all sources are relative.
     - sourcesContent: Optional. An array of contents of the original
                       source files.
     - mappings: A string of base64 VLQs which contain the actual mappings.
     - file: The generated file this source map is associated with.

   Here is an example source map, taken from the source map spec[0]:

       {
         version : 3,
         file: "out.js",
         sourceRoot : "",
         sources: ["foo.js", "bar.js"],
         names: ["src", "maps", "are", "fun"],
         mappings: "AA,AB;;ABCDE;"
       }

   [0]: https://docs.google.com/document/d/
        1U1RGAehQwRypUTovF1KRlpiOFze0b-_2gc6fAH0KY0k/edit?pli=1#
  '''

  #-----------------------------------------------------------------
  # The version of the source mapping spec that we are consuming.
  _version = 3

  #-----------------------------------------------------------------
  def __init__(self, aSourceMap, dontInitialize=False):
    '''Creates a new SourceMapConsumer.
    '''

    if dontInitialize:
      return

    sourceMap = aSourceMap

    version = sourceMap['version']
    sources = sourceMap['sources']
    # Sass 3.3 leaves out the 'names' array, so we deviate from the spec (which
    # requires the array) to play nice here.
    names = sourceMap.get('names', [])
    sourceRoot = sourceMap.get('sourceRoot', None)
    sourcesContent = sourceMap.get('sourcesContent', None)
    mappings = sourceMap['mappings']
    file = sourceMap.get('file', None)

    # Once again, Sass deviates from the spec and supplies the version as a
    # string rather than a number, so we use loose equality checking here.
    if version != SourceMapConsumer._version:
      raise SourceMapException('Unsupported version: ' + version)

    # Pass `true` below to allow duplicate names and sources. While source maps
    # are intended to be compressed and deduplicated, the TypeScript compiler
    # sometimes generates source maps with duplicates in them. See Github issue
    # #72 and bugzil.la/889492.
    #
    # Comment out by MPB.
    #this._names = ArraySet.fromArray(names, true);
    #this._sources = ArraySet.fromArray(sources, true);

    self._names = names
    self._sources = sources

    self.sourceRoot = sourceRoot
    self.sourcesContent = sourcesContent
    self._mappings = mappings
    self.file = file

    # pylint: disable=C0103
    self.__generatedMappings = None
    self.__originalMappings = None

  #-----------------------------------------------------------------
  @staticmethod
  def fromSourceMap(aSourceMap):
    ''' Create a SourceMapConsumer from a SourceMapGenerator.

    @param SourceMapGenerator aSourceMap
           The source map that will be consumed.
    @returns SourceMapConsumer
    '''

    smc = SourceMapConsumer(None, True)

    # pylint: disable=W0212
    smc._names = aSourceMap._names[:]
    smc._sources = aSourceMap._sources[:]
    smc.sourceRoot = aSourceMap._sourceRoot
    smc.sourcesContent = aSourceMap._generateSourcesContent(smc._sources[:],
                                                            smc.sourceRoot)
    smc.file = aSourceMap._file

    smc.__generatedMappings = aSourceMap._mappings[:]
    theKey = SourceMapping.compareByGeneratedPositionsKey
    smc.__generatedMappings.sort(key=theKey)

    smc.__originalMappings = aSourceMap._mappings[:]
    theKey = SourceMapping.compareByOriginalPositionsKey
    smc.__originalMappings.sort(key=theKey)

    return smc

  #-----------------------------------------------------------------
  @property
  def sources(self):
    ''' The list of original sources.
    '''

    ret = []
    for src in self._sources:
      if self.sourceRoot:
        src = os.path.join(self.sourceRoot, src)
      ret.append(src)

    return ret


  # `__generatedMappings` and `__originalMappings` are arrays that hold the
  # parsed mapping coordinates from the source map's "mappings" attribute. They
  # are lazily instantiated, accessed via the `_generatedMappings` and
  # `_originalMappings` getters respectively, and we only parse the mappings
  # and create these arrays once queried for a source location. We jump through
  # these hoops because there can be many thousands of mappings, and parsing
  # them is expensive, so we only want to do it if we must.
  #
  # Each object in the arrays is of the form:
  #
  #     {
  #       generatedLine: The line number in the generated code,
  #       generatedColumn: The column number in the generated code,
  #       source: The path to the original source file that generated this
  #               chunk of code,
  #       originalLine: The line number in the original source that
  #                     corresponds to this chunk of generated code,
  #       originalColumn: The column number in the original source that
  #                       corresponds to this chunk of generated code,
  #       name: The name of the original symbol which generated this chunk of
  #             code.
  #     }
  #
  # All properties except for `generatedLine` and `generatedColumn` can be
  # `null`.
  #
  # `_generatedMappings` is ordered by the generated positions.
  #
  # `_originalMappings` is ordered by the original positions.

  #-----------------------------------------------------------------
  @property
  def _generatedMappings(self):

    if self.__generatedMappings is None:
      self.__generatedMappings = []
      self.__originalMappings = []
      self._parseMappings(self._mappings, self.sourceRoot)

    return self.__generatedMappings


  #-----------------------------------------------------------------
  @property
  def _originalMappings(self):

    if self.__originalMappings is None:
      self.__generatedMappings = []
      self.__originalMappings = []
      self._parseMappings(self._mappings, self.sourceRoot)

    return self.__originalMappings

  #-----------------------------------------------------------------
  # pylint: disable=W0613
  def _parseMappings(self, aStr, aSourceRoot):
    '''Parse the mappings in a string in to a data structure which we can easily
    query (the ordered arrays in the `self.__generatedMappings` and
    `self.__originalMappings` properties).
    '''

    generatedLine = 1
    previousGeneratedColumn = 0
    previousOriginalLine = 0
    previousOriginalColumn = 0
    previousSource = 0
    previousName = 0
    mappingSeparators = ',;'
    theStr = aStr

    while theStr:
      if theStr[0] == ';':
        generatedLine += 1
        theStr = theStr[1:]
        previousGeneratedColumn = 0

      elif theStr[0] == ',':
        theStr = theStr[1:]

      else:
        mapping = SourceMapping(None, None, None, None, None, None)
        mapping.generatedLine = generatedLine

        # Generated column.
        temp = VlqBase64.decode(theStr)
        mapping.generatedColumn = previousGeneratedColumn + temp.value
        previousGeneratedColumn = mapping.generatedColumn
        theStr = temp.rest

        if theStr and theStr[0] not in mappingSeparators:
          # Original source.
          temp = VlqBase64.decode(theStr)
          mapping.source = self._sources[previousSource + temp.value]
          previousSource += temp.value
          theStr = temp.rest
          if not theStr or theStr[0] in mappingSeparators:
            raise SourceMapException('Found a source, but no line and column')

          # Original line.
          temp = VlqBase64.decode(theStr)
          mapping.originalLine = previousOriginalLine + temp.value
          previousOriginalLine = mapping.originalLine
          # Lines are stored 0-based
          mapping.originalLine += 1
          theStr = temp.rest
          if not theStr or theStr[0] in mappingSeparators:
            raise SourceMapException('Found a source and line, but no column')

          # Original column.
          temp = VlqBase64.decode(theStr)
          mapping.originalColumn = previousOriginalColumn + temp.value
          previousOriginalColumn = mapping.originalColumn
          theStr = temp.rest

          if theStr and theStr[0] not in mappingSeparators:

            # Original name.
            temp = VlqBase64.decode(theStr)
            mapping.name = self._names[previousName + temp.value]
            previousName += temp.value
            theStr = temp.rest

        self.__generatedMappings.append(mapping)
        if mapping.originalLine.__class__ == int:
          self.__originalMappings.append(mapping)

    theKey = SourceMapping.compareByOriginalPositionsKey
    self.__originalMappings.sort(key=theKey)

  #-----------------------------------------------------------------
  def _findMapping(self, aNeedle, aMappings, aLineName,
                   aColumnName, aComparator):
    '''Find the mapping that best matches the hypothetical "needle" mapping that
    we are searching for in the given "haystack" of mappings.
    '''

    # To return the position we are searching for, we must first find the
    # mapping for the given position and then return the opposite position it
    # points to. Because the mappings are sorted, we can use binary search to
    # find the best mapping.

    if getattr(aNeedle, aLineName) <= 0:
      raise SourceMapException('Line must be greater than or equal to 1, got '
                          + str(getattr(aNeedle, aLineName)))
    if getattr(aNeedle, aColumnName) < 0:
      raise SourceMapException('Column must be greater than or equal to 0, got '
                          + str(getattr(aNeedle, aColumnName)))

    return JsUtil.search(aNeedle, aMappings, aComparator)

  #-----------------------------------------------------------------
  def originalPositionFor(self, lineColumn):
    '''Returns the original source, line, and column information for the
    generated
    source's line and column positions provided. The only argument is an object
    with the following properties:

      - line: The line number in the generated source.
      - column: The column number in the generated source.

    and an object is returned with the following properties:

      - source: The original source file, or null.
      - line: The line number in the original source, or null.
      - column: The column number in the original source, or null.
      - name: The original identifier, or null.
    '''

    needle = SourceMapping(generatedLine=lineColumn.line,
                           generatedColumn=lineColumn.column,
                           originalLine=None,
                           originalColumn=None,
                           source=None,
                           name=None)

    mapping = self._findMapping(needle,
                                self._generatedMappings,
                                "generatedLine",
                                "generatedColumn",
                                SourceMapping.compareByGeneratedPositions)

    if mapping:
      source = getattr(mapping, 'source', None)
      if source and self.sourceRoot:
        source = os.path.join(self.sourceRoot, source)

      return SourceLineColumn(source,
                              mapping.originalLine,
                              mapping.originalColumn,
                              mapping.name)

    return SourceLineColumn(None, None, None, None)

  #-----------------------------------------------------------------
  def sourceContentFor(self, aSource):
    '''Returns the original source content. The only argument is the url of the
    original source file. Returns null if no original source content is
    availible.
    '''

    if not self.sourcesContent:
      return None

    if self.sourceRoot:
      aSource = JsUtil.relpath(aSource, self.sourceRoot)

    if aSource in self._sources:
      return self.sourcesContent[self._sources.index(aSource)]

    if self.sourceRoot and JsUtil.urlRe.match(self.sourceRoot):
      url = urllib.parse.urlparse(self.sourceRoot)

      # file:// URIs and absolute paths lead to unexpected behavior for
      # many users. We can help them out when they expect file:// URIs to
      # behave like it would if they were running a local HTTP server. See
      # https://bugzilla.mozilla.org/show_bug.cgi?id=885597.
      fileUriAbsPath = aSource.replace('file://', "")
      if (url.scheme == "file" and
          fileUriAbsPath in self._sources):
        return self.sourcesContent[self._sources.index(fileUriAbsPath)]

      if ((not url.path or url.path == "/") and
          ("/" + aSource) in self._sources):
        return self.sourcesContent[self._sources.index("/" + aSource)]

    if self.sourceRoot.startswith('file:'):
      raise SourceMapException('file URL code not ported yet.')

    raise SourceMapException('"' + aSource + '" is not in the SourceMap.')

  #-----------------------------------------------------------------
  def generatedPositionFor(self, sourceLineColumn):
    '''Returns the generated line and column information for the original
    source,
    line, and column positions provided. The only argument is an object with
    the following properties:

      - source: The filename of the original source.
      - line: The line number in the original source.
      - column: The column number in the original source.

    and an object is returned with the following properties:

      - line: The line number in the generated source, or null.
      - column: The column number in the generated source, or null.
    '''

    needle = SourceMapping(generatedLine=None,
                           generatedColumn=None,
                           originalLine=sourceLineColumn.line,
                           originalColumn=sourceLineColumn.column,
                           source=sourceLineColumn.source,
                           name=None)

    if self.sourceRoot:
      needle.source = JsUtil.relpath(needle.source, self.sourceRoot)

    mapping = self._findMapping(needle,
                                self._originalMappings,
                                "originalLine",
                                "originalColumn",
                                SourceMapping.compareByOriginalPositions)

    if mapping:
      return LineColumn(mapping.generatedLine, mapping.generatedColumn)

    return LineColumn(None, None)

  #-----------------------------------------------------------------
  GENERATED_ORDER = 1

  #-----------------------------------------------------------------
  ORIGINAL_ORDER = 2

  #-----------------------------------------------------------------
  def eachMapping(self, aCallback, aOrder=GENERATED_ORDER):
    '''Iterate over each mapping between an original source/line/column and a
    generated line/column in this source map.

    @param Function aCallback
           The function that is called with each mapping.
    @param aOrder
           Either `SourceMapConsumer.GENERATED_ORDER` or
           `SourceMapConsumer.ORIGINAL_ORDER`. Specifies whether you want to
           iterate over the mappings sorted by the generated file's line/column
           order or the original's source/line/column order, respectively.
           Defaults to
           `SourceMapConsumer.GENERATED_ORDER`.
    '''

    newMappings = self.addRootToMappings(aOrder)

    for mapping in newMappings:
      aCallback(mapping)

  #-----------------------------------------------------------------
  def addRootToMappings(self, aOrder=GENERATED_ORDER):

    order = aOrder

    if order == SourceMapConsumer.GENERATED_ORDER:
      mappings = self._generatedMappings
    elif order == SourceMapConsumer.ORIGINAL_ORDER:
      mappings = self._originalMappings
    else:
      raise SourceMapException("Unknown order of iteration:" + str(order))

    sourceRoot = self.sourceRoot

    newMappings = []

    for mapping in mappings:

      source = mapping.source
      if source and sourceRoot:
        source = os.path.join(sourceRoot, source)

      m = SourceMapping(source=source,
                        generatedLine=mapping.generatedLine,
                        generatedColumn=mapping.generatedColumn,
                        originalLine=mapping.originalLine,
                        originalColumn=mapping.originalColumn,
                        name=mapping.name)

      newMappings.append(m)

    return newMappings
