#-------------------------------------------------------------------
#  JsUtil.py
#
#  The JsUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Utility javascript functions.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import re
import urllib.parse
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def addUnique(aList, item):
  '''Adds an item to a list if it isn't already in the list.
  '''

  if item not in aList:
    aList.append(item)

#-------------------------------------------------------------------
def toSetString(key):
  # In the original javascript implementation, a prefix was appended
  # to map keys to prevent clashes with built-in Object methods.
  return key

#-------------------------------------------------------------------
def fromSetString(key):
  # In the original javascript implementation, a prefix was appended
  # to map keys to prevent clashes with built-in Object methods.
  return key

#-------------------------------------------------------------------
urlReStr = r'([\w+\-.]+):\/\/((\w+:\w+)@)?([\w.]+)?(:(\d+))?(\S+)?'
urlRe = re.compile(urlReStr)

#-------------------------------------------------------------------
def relpath(fileName, rootPath):
  '''Returns the file name relative to the root path.

  Like os.path.relpath(), but it won't return the name relative to the
  current directory.  Instead, just returns the file name unchanged
  if it doesn't appear to be under the root path.
  '''

  if rootPath.endswith('/'):
    rootPath = rootPath[:-1]

  if urlRe.match(rootPath):
    url = urllib.parse.urlparse(rootPath)
    if fileName.startswith('/') and url.path == '/':
      return fileName[1:]

  if fileName.startswith(rootPath + '/'):
    return fileName[len(rootPath) + 1:]
  else:
    return fileName

#-------------------------------------------------------------------
def recursiveSearch(aLow, aHigh, aNeedle, aHaystack, aCompare):
  '''Recursive implementation of binary search.

  @param aLow Indices here and lower do not contain the needle.
  @param aHigh Indices here and higher do not contain the needle.
  @param aNeedle The element being searched for.
  @param aHaystack The non-empty array being searched.
  @param aCompare Function which takes two elements and returns -1, 0, or 1.
  '''

  # This function terminates when one of the following is true:
  #
  #   1. We find the exact element we are looking for.
  #
  #   2. We did not find the exact element, but we can return the next
  #      closest element that is less than that element.
  #
  #   3. We did not find the exact element, and there is no next-closest
  #      element which is less than the one we are searching for, so we
  #      return null.
  mid = int((aHigh - aLow) / 2) + aLow
  cmpValue = aCompare(aNeedle, aHaystack[mid], True)
  if cmpValue == 0:
    # Found the element we are looking for.
    return aHaystack[mid]

  elif cmpValue > 0:
    # aHaystack[mid] is greater than our needle.
    if aHigh - mid > 1:
      # The element is in the upper half.
      return recursiveSearch(mid, aHigh, aNeedle, aHaystack, aCompare)

    # We did not find an exact match, return the next closest one
    # (termination case 2).
    return aHaystack[mid]

  else:
    # aHaystack[mid] is less than our needle.
    if mid - aLow > 1:
      # The element is in the lower half.
      return recursiveSearch(aLow, mid, aNeedle, aHaystack, aCompare)

    # The exact needle element was not found in this haystack. Determine if
    # we are in termination case (2) or (3) and return the appropriate thing.
    if aLow < 0:
      return None
    else:
      return aHaystack[aLow]

#-------------------------------------------------------------------
def search(aNeedle, aHaystack, aCompare):
  '''This is an implementation of binary search which will always try and return
  the next lowest value checked if there is no exact hit. This is because
  mappings between original and generated line/col pairs are single points,
  and there is an implicit region between each of them, so a miss just means
  that you aren't on the very start of a region.

  @param aNeedle The element you are looking for.
  @param aHaystack The array that is being searched.
  @param aCompare A function which takes the needle and an element in the
      array and returns -1, 0, or 1 depending on whether the needle is less
      than, equal to, or greater than the element, respectively.
  '''

  # pylint: disable=C1801

  if len(aHaystack) > 0:
    return recursiveSearch(-1, len(aHaystack), aNeedle, aHaystack, aCompare)
  else:
    return None
