#-------------------------------------------------------------------
#  TestColorBinMap.py
#
#  The TestColorBinMap module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the ColorBinMap class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.db.postgis import ColorBinMap
from ai.axe.build.unittest import AxeSimpleTestCase
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestColorBinMap(AxeSimpleTestCase):
  '''Unit test for the ColorBinMap class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testCreation(self) -> None:
    '''Test creating a color map.
    '''

    theMap = ColorBinMap((0, 0, 0, 0), (1, 1, 1, 255))

    theMap.addBin(3, True, (2, 2, 2, 255))
    theMap.addBin(7, False, (3, 3, 3, 255))
    theMap.addBin(10, True, (4, 4, 4, 255))

    # Exception if attempt to add lower value.
    try:
      theMap.addBin(9, True, (5, 5, 5, 255))
      self.fail("No exception for lower value")
    except ValueError:
      pass

    self.assertEqual((0, 0, 0, 0), theMap.noDataColor, "noDataColor")

    expected1 = [(x, x, x, 255) for x in range(1, 5)]
    self.assertEqual(expected1, theMap.colors, "colors")

    expected2 = [3, 7, 10]
    self.assertEqual(expected2, theMap.borders, "borders")

    expected3 = [True, False, True]
    self.assertEqual(expected3, theMap.borderGoesHighs, "borderGoesHighs")

    # Make sure no exception is thrown.
    str(theMap)

  #-----------------------------------------------------------------
  def testColorForValue(self) -> None:
    '''Test converting values to colors.
    '''

    theMap = ColorBinMap((0, 0, 0, 0), (1, 1, 1, 255))

    theMap.addBin(3, True, (2, 2, 2, 255))
    theMap.addBin(7, False, (3, 3, 3, 255))
    theMap.addBin(10, True, (4, 4, 4, 255))

    self.assertEqual((0, 0, 0, 0), theMap.colorForValue(None), "nodata")

    self.assertEqual((1, 1, 1, 255), theMap.colorForValue(2), "lowest")
    self.assertEqual((2, 2, 2, 255), theMap.colorForValue(3), "bin goes up")
    self.assertEqual((2, 2, 2, 255), theMap.colorForValue(7), "bin goes down")
    self.assertEqual((3, 3, 3, 255), theMap.colorForValue(7.01), "middle")
    self.assertEqual((4, 4, 4, 255), theMap.colorForValue(10), "highest equal")
    self.assertEqual((4, 4, 4, 255), theMap.colorForValue(10.01), "highest")

  #-----------------------------------------------------------------
  def testBinSql(self) -> None:
    '''Test generating bin SQL.
    '''

    theMap = ColorBinMap((0, 0, 0, 0), (1, 1, 1, 255))

    theMap.addBin(3, True, (2, 2, 2, 255))
    theMap.addBin(7, False, (3, 3, 3, 255))
    theMap.addBin(10, True, (4, 4, 4, 255))

    expected = (
      'case ' +
      'when [rast.val] < 3 then 1 ' +
      'when [rast.val] >= 3 and [rast.val] <= 7 then 2 ' +
      'when [rast.val] > 7 and [rast.val] < 10 then 3 ' +
      'when [rast.val] >= 10 then 4 ' +
      'end'
    )

    self.assertEqual(expected, theMap.getBinSql(), "binSql")

  #-----------------------------------------------------------------
  def testGdalColorMap(self) -> None:
    '''Test generating a GDAL color map.
    '''

    theMap = ColorBinMap((0, 0, 0, 0), (1, 1, 1, 255))

    theMap.addBin(3, True, (2, 2, 2, 255))
    theMap.addBin(7, False, (3, 3, 3, 255))
    theMap.addBin(10, True, (4, 4, 4, 255))

    expected = (
      '4 4 4 4 255\n' +
      '3 3 3 3 255\n' +
      '2 2 2 2 255\n' +
      '1 1 1 1 255\n' +
      '0 0 0 0 0'
    )

    self.assertEqual(expected, theMap.getGdalColorMap(), "color map")
