#-------------------------------------------------------------------
#  SingleMultiPolygonType.py
#
#  The SingleMultiPolygonType class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the SingleMultiPolygonType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.wkt import Polygon
from ai.axe.wkt import MultiPolygon
from ai.axe.wkt import WktReader
from ai.axe.wkt import WktWriter
from sqlalchemy import func
from sqlalchemy.types import UserDefinedType
#
# Import statements go above this line.
#-------------------------------------------------------------------

# pylint: disable = C0103, super-init-not-called

#===================================================================
class SingleMultiPolygonType(UserDefinedType):
  '''A Polygon/MultiPolygon stored in a PostGIS geometry(Polygon, 4326) column.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new SingleMultiPolygonType.
    '''

    pass

  #-----------------------------------------------------------------
  @property
  def python_type(self):

    # There's no single Python type for this.
    # It can be either a Polygon or Multipolygon.
    # Since we can't truthfully return anything, just raise an exception.

    return UserDefinedType.python_type(self)

  #-----------------------------------------------------------------
  def get_col_spec(self):
    return 'geometry(geometry, 4326)'

  #-----------------------------------------------------------------
  def column_expression(self, colexpr):
    return func.ST_AsText(colexpr, type_=self)

  #-----------------------------------------------------------------
  def result_processor(self, dialect, coltype):
    '''Returns the processor function to use to convert from SQL to Python.
    '''

    return SingleMultiPolygonType.stringToObject

  #-----------------------------------------------------------------
  @staticmethod
  def stringToObject(value):
    '''Processes a selected DB value into a Polygon object.
    '''

    if value is None:
      return None
    else:
      reader = WktReader()
      return reader.read(value)

  #-----------------------------------------------------------------
  def bind_expression(self, bindvalue):
    return func.ST_GeomFromText(bindvalue, 4326, type_=self)

  #-----------------------------------------------------------------
  def bind_processor(self, dialect):
    '''Returns the processor function to use to convert from Python to SQL.
    '''

    return SingleMultiPolygonType.objectToString

  #-----------------------------------------------------------------
  @staticmethod
  def objectToString(polygon):

    if polygon is None:
      return None

    if not isinstance(polygon, (Polygon, MultiPolygon)):

      msg = "Expected axe.wkt.Polygon or MultiPolygon, got: "
      msg += str(polygon.__class__)
      raise TypeError(msg)

    else:

      writer = WktWriter()
      return writer.write(polygon)
