#-------------------------------------------------------------------
#  PolygonType.py
#
#  The PolygonType class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the PolygonType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.wkt import Polygon
from ai.axe.wkt import WktReader
from ai.axe.wkt import WktWriter
from sqlalchemy import func
from sqlalchemy.types import UserDefinedType
#
# Import statements go above this line.
#-------------------------------------------------------------------

# pylint: disable = C0103, super-init-not-called

#===================================================================
class PolygonType(UserDefinedType):
  '''A Polygon stored in a PostGIS geometry(Polygon, 4326) column.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new PolygonType.
    '''

    pass

  #-----------------------------------------------------------------
  @property
  def python_type(self):
    return Polygon

  #-----------------------------------------------------------------
  def get_col_spec(self):
    return 'geometry(polygon, 4326)'

  #-----------------------------------------------------------------
  def column_expression(self, colexpr):
    return func.ST_AsText(colexpr, type_=self)

  #-----------------------------------------------------------------
  def result_processor(self, dialect, coltype):
    '''Returns the processor function to use to convert from SQL to Python.
    '''

    return PolygonType.stringToObject

  #-----------------------------------------------------------------
  @staticmethod
  def stringToObject(value):
    '''Processes a selected DB value into a Polygon object.
    '''

    if value is None:
      return None
    else:
      reader = WktReader()
      return reader.read(value)

  #-----------------------------------------------------------------
  def bind_expression(self, bindvalue):
    return func.ST_GeomFromText(bindvalue, 4326, type_=self)

  #-----------------------------------------------------------------
  def bind_processor(self, dialect):
    '''Returns the processor function to use to convert from Python to SQL.
    '''

    return PolygonType.objectToString

  #-----------------------------------------------------------------
  @staticmethod
  def objectToString(polygon):

    if polygon is None:
      return None

    if not isinstance(polygon, Polygon):
      msg = "Expected axe.wkt.Polygon, got: " + str(polygon.__class__)
      raise TypeError(msg)

    else:

      writer = WktWriter()
      return writer.write(polygon)
