#-------------------------------------------------------------------
#  PointType.py
#
#  The PointType class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the PointType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.wkt import Point
from ai.axe.wkt import WktReader
from sqlalchemy import func
from sqlalchemy.types import UserDefinedType
#
# Import statements go above this line.
#-------------------------------------------------------------------

# pylint: disable = C0103, super-init-not-called

#===================================================================
class PointType(UserDefinedType):
  '''An (x, y) point stored in a PostGIS geometry(Point, 4326) column.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new PointType.
    '''

    pass

  #-----------------------------------------------------------------
  @property
  def python_type(self):
    return Point

  #-----------------------------------------------------------------
  def get_col_spec(self):
    return 'geometry(point, 4326)'

  #-----------------------------------------------------------------
  def column_expression(self, colexpr):
    return func.ST_AsText(colexpr, type_=self)

  #-----------------------------------------------------------------
  def result_processor(self, dialect, coltype):
    '''Returns the processor function to use to convert from SQL to Python.
    '''

    return PointType.stringToObject

  #-----------------------------------------------------------------
  @staticmethod
  def stringToObject(value):
    '''Processes a selected DB value into a Point object.
    '''

    if value is None:
      return None
    else:
      reader = WktReader()
      return reader.read(value)

  #-----------------------------------------------------------------
  def bind_expression(self, bindvalue):
    return func.ST_GeomFromText(bindvalue, 4326, type_=self)

  #-----------------------------------------------------------------
  def bind_processor(self, dialect):
    '''Returns the processor function to use to convert from Python to SQL.
    '''

    return PointType.objectToString

  #-----------------------------------------------------------------
  @staticmethod
  def objectToString(point):

    if point is None:
      return None

    elif not isinstance(point, Point):
      msg = "Expected axe.wkt.Point, got: " + str(point.__class__)
      raise TypeError(msg)

    else:
      return "POINT(%s %s)" % (point.x, point.y)
