#-------------------------------------------------------------------
#  Point3dType.py
#
#  The Point3dType class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Point3dType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.wkt import Point3d
from ai.axe.wkt import WktReader
from sqlalchemy import func
from sqlalchemy.types import UserDefinedType
#
# Import statements go above this line.
#-------------------------------------------------------------------

# pylint: disable = C0103, super-init-not-called

#===================================================================
class Point3dType(UserDefinedType):
  '''An (x, y, z) point stored in a PostGIS geometry(PointZ, 4326) column.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new Point3dType.
    '''

    pass

  #-----------------------------------------------------------------
  @property
  def python_type(self):
    return Point3d

  #-----------------------------------------------------------------
  def get_col_spec(self):
    return 'geometry(pointz, 4326)'

  #-----------------------------------------------------------------
  def column_expression(self, colexpr):
    return func.ST_AsText(colexpr, type_=self)

  #-----------------------------------------------------------------
  def result_processor(self, dialect, coltype):
    '''Returns the processor function to use to convert from SQL to Python.
    '''

    return Point3dType.stringToObject

  #-----------------------------------------------------------------
  @staticmethod
  def stringToObject(value):
    '''Processes a selected DB value into a Point object.
    '''

    if value is None:
      return None
    else:
      reader = WktReader()
      return reader.read(value)

  #-----------------------------------------------------------------
  def bind_expression(self, bindvalue):
    return func.ST_GeomFromText(bindvalue, 4326, type_=self)

  #-----------------------------------------------------------------
  def bind_processor(self, dialect):
    '''Returns the processor function to use to convert from Python to SQL.
    '''

    return Point3dType.objectToString

  #-----------------------------------------------------------------
  @staticmethod
  def objectToString(point):

    if point is None:
      return None

    elif not isinstance(point, Point3d):
      msg = "Expected axe.wkt.Point3d, got: " + str(point.__class__)
      raise TypeError(msg)

    else:
      return "POINT Z(%s %s %s)" % (point.x, point.y, point.z)
