#-------------------------------------------------------------------
#  MultiPolygonType.py
#
#  The MultiPolygonType class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the MultiPolygonType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.wkt import MultiPolygon
from ai.axe.wkt import WktReader
from ai.axe.wkt import WktWriter
from sqlalchemy import func
from sqlalchemy.types import UserDefinedType
#
# Import statements go above this line.
#-------------------------------------------------------------------

# pylint: disable = C0103, super-init-not-called

#===================================================================
class MultiPolygonType(UserDefinedType):
  '''A MultiPolygon stored in a PostGIS geometry(MultiPolygon, 4326) column.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new MultiPolygonType.
    '''

    pass

  #-----------------------------------------------------------------
  @property
  def python_type(self):
    return MultiPolygon

  #-----------------------------------------------------------------
  def get_col_spec(self):
    return 'geometry(multipolygon, 4326)'

  #-----------------------------------------------------------------
  def column_expression(self, colexpr):
    return func.ST_AsText(colexpr, type_=self)

  #-----------------------------------------------------------------
  def result_processor(self, dialect, coltype):
    '''Returns the processor function to use to convert from SQL to Python.
    '''

    return MultiPolygonType.stringToObject

  #-----------------------------------------------------------------
  @staticmethod
  def stringToObject(value):
    '''Processes a selected DB value into a MultiPolygon object.
    '''

    if value is None:
      return None
    else:
      reader = WktReader()
      return reader.read(value)

  #-----------------------------------------------------------------
  def bind_expression(self, bindvalue):
    return func.ST_GeomFromText(bindvalue, 4326, type_=self)

  #-----------------------------------------------------------------
  def bind_processor(self, dialect):
    '''Returns the processor function to use to convert from Python to SQL.
    '''

    return MultiPolygonType.objectToString

  #-----------------------------------------------------------------
  @staticmethod
  def objectToString(polygon):

    if polygon is None:
      return None

    if not isinstance(polygon, MultiPolygon):
      msg = "Expected axe.wkt.MultiPolygon, got: " + str(polygon.__class__)
      raise TypeError(msg)

    else:

      writer = WktWriter()
      return writer.write(polygon)
