#-------------------------------------------------------------------
#  TestDbListProxy.py
#
#  The TestDbListProxy module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the DbListProxy class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.db.alchemy import DbListProxy
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import Any
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestDbListProxy(AxeSimpleTestCase):
  '''Unit test for the DbListProxy class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testList(self) -> None:
    '''Test the list methods.
    '''

    class ListItemObject:

      def __init__(self) -> None:

        self.dbListPosition: int = -1
        self.bar: int = 0

    srcList: List[ListItemObject] = []

    proxy: DbListProxy[int] = DbListProxy(ListItemObject, 'bar', srcList)

    # Test __len__.

    self.assertEqual(0, len(proxy), 'len')

    # Test append.

    proxy.append(3)
    proxy.append(4)
    proxy.append(5)

    self.assertEqual(3, len(proxy), 'len')
    self.assertEqual(3, len(srcList), 'len')

    self.assertEqual(3, srcList[0].bar, 'append 0')
    self.assertEqual(4, srcList[1].bar, 'append 1')
    self.assertEqual(5, srcList[2].bar, 'append 2')

    self.assertDbListPosition(srcList, 'append')

    # Test get item.

    self.assertEqual(4, proxy[1], 'get item')

    # Test set item.

    proxy[1] = 6
    self.assertEqual(6, srcList[1].bar, 'set item')

    # Test delete item.

    del proxy[1]
    self.assertEqual(2, len(srcList), 'delete len')
    self.assertEqual(3, srcList[0].bar, 'delete 0')
    self.assertEqual(5, srcList[1].bar, 'delete 1')

    # Test contains.

    self.assertEqual(True, 3 in proxy, 'contains')
    self.assertEqual(False, 4 in proxy, 'contains')

    # Test get and set slice.

    proxy[:] = []
    self.assertEqual(0, len(srcList), 'delete via set slice')

    proxy.extend([2, 3, 4])

    self.assertEqual([3, 4], proxy[1:3], 'get slice')
    self.assertEqual([4, 2], proxy[2::-2], 'get slice')

    proxy[1:3] = [5, 6]

    self.assertEqual([2, 5, 6], list(proxy), 'set slice')

    del proxy[1:3]
    self.assertEqual([2], list(proxy), 'del slice')

    proxy[:] = [11, 12, 13, 14]
    self.assertEqual([11, 12, 13, 14], list(proxy), 'set all via slice')

    proxy[:] = [11, 12, 13, 14]
    proxy[0:3] = [1, 2, 3]
    self.assertEqual([1, 2, 3, 14], list(proxy), 'set slice that grows list')

    proxy[:] = [11, 12, 13, 14]
    proxy[1:3] = [1, 2, 3]
    expected = [11, 1, 2, 3, 14]
    self.assertEqual(expected, list(proxy), 'set slice that grows list')

    proxy[:] = [11, 12, 13, 14]
    proxy[1:3] = [1]
    self.assertEqual([11, 1, 14], list(proxy), 'set slice that shrinks list')

    proxy[:] = [1, 2, 3, 4, 5, 6, 7]
    proxy[2:7:2] = [11, 12, 13]
    expected = [1, 2, 11, 4, 12, 6, 13]
    self.assertEqual(expected, list(proxy), 'set slice with skip')

    proxy[:] = [1, 2, 3, 4, 5, 6, 7]
    try:
      proxy[2:7:2] = [11, 12]
      self.fail("No exception for set slice with skip and wrong size.")
    except ValueError as ex:
      msg = 'to assign sequence of size 2 to extended slice of size 3'
      self.assertTrue(msg in str(ex), 'set slice with skip and wrong size')

    # Iter

    del proxy[:]
    proxy.extend([1, 2, 3])

    self.assertEqual([1, 2, 3], list(iter(proxy)), 'iter')

    # Count.

    self.assertEqual(1, proxy.count(2), 'count')

    # Insert.

    proxy.insert(1, 4)
    self.assertEqual([1, 4, 2, 3], list(proxy), 'insert')

    # Pop.

    proxy.pop()
    self.assertEqual([1, 4, 2], list(proxy), 'pop')

    # Remove.

    proxy.remove(4)
    self.assertEqual([1, 2], list(proxy), 'remove')

    try:
      proxy.remove(4)
      self.fail("No exception for bad remove value.")
    except ValueError:
      pass

    # Not implemented.

    try:
      proxy.reverse()
      self.fail("No exception for not implemented.")
    except TypeError:
      pass

    try:
      proxy.sort()
      self.fail("No exception for not implemented.")
    except TypeError:
      pass

    # Clear.

    proxy.clear()
    self.assertEqual([], list(proxy), 'clear')

    # Copy.

    proxy.extend([1, 2, 3])
    self.assertEqual([1, 2, 3], proxy.copy(), 'copy')

    # Compare.

    self.assertEqual(True, proxy == [1, 2, 3])
    self.assertEqual(True, proxy != [1, 2, 4])
    self.assertEqual(True, proxy < [4])
    self.assertEqual(True, proxy <= [4])
    self.assertEqual(True, proxy > [1])
    self.assertEqual(True, proxy >= [1])

    # Add.

    self.assertEqual([1, 2, 3, 4], proxy + [4], 'add')
    self.assertEqual([4, 1, 2, 3], [4] + proxy, 'add')

    proxy += [4]
    self.assertEqual([1, 2, 3, 4], proxy, 'add')
    proxy.pop()

    # Multiply.

    self.assertEqual([1, 2, 3, 1, 2, 3], proxy * 2, 'multiply')
    self.assertEqual([1, 2, 3, 1, 2, 3], 2 * proxy, 'multiply')

    proxy *= 2
    self.assertEqual([1, 2, 3, 1, 2, 3], proxy, 'multiply')

    proxy *= 0
    self.assertEqual([], proxy, 'multiply')

    # String.

    proxy.clear()
    proxy.extend([1, 2, 3])
    self.assertEqual('[1, 2, 3]', str(proxy), 'str')

    # Exceptions.

    # pylint: disable=W0104

    try:
      proxy * 'bad type'               # type: ignore
      self.fail("No exception.")
    except TypeError:
      pass

    try:
      proxy *= 'bad type'              # type: ignore
      self.fail("No exception.")
    except TypeError:
      pass

    try:
      hash(proxy)
      self.fail("No exception.")
    except TypeError:
      pass

    try:
      proxy += 3                       # type: ignore
      self.fail("No exception.")
    except TypeError:
      pass

    try:
      proxy + 3                        # type: ignore
      self.fail("No exception.")
    except TypeError:
      pass

    try:
      3 + proxy                        # type: ignore
      self.fail("No exception.")
    except TypeError as ex:
      pass

  #-----------------------------------------------------------------
  def assertDbListPosition(self, srcList: List[Any], msg: str) -> None:
    '''Checks that dbListPosition is correct.

    Each item in the list must have a dbListPostition, and it must
    match its index in the list.
    '''

    msg += ": list position"

    for i, item in enumerate(srcList):
      self.assertEqual(i, item.dbListPosition, msg)
