#-------------------------------------------------------------------
#  TestDbDictProxy.py
#
#  The TestDbDictProxy module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the DbDictProxy class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.db.alchemy import DbDictProxy
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import Any
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestDbDictProxy(AxeSimpleTestCase):
  '''Unit test for the DbDictProxy class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testProxy(self) -> None:
    '''Test the dict proxy.
    '''

    class DictItem:
      def __init__(self) -> None:
        self.dbDictPosition: int = -1
        self.key: str = 'not set yet'
        self.value: int = -1

      def __repr__(self) -> str:
        return 'Item: %s = %s' % (self.key, self.value)

    srcList: List[DictItem] = []

    proxy: DbDictProxy[str,
                       int] = DbDictProxy(DictItem, 'key', 'value', srcList)

    self.assertEqual(0, len(proxy), 'empty list')

    proxy['foo'] = 30
    proxy['bar'] = 31

    self.assertEqual({'foo': 30, 'bar': 31}, proxy, 'setitem')
    self.assertDbDictPosition(srcList, 'setitem')

    self.assertEqual({'foo': 30, 'bar': 31}, proxy.copy(), 'copy')

    self.assertEqual(31, proxy['bar'], 'getitem')
    self.assertEqual(30, proxy['foo'], 'getitem')

    self.assertEqual(31, proxy.get('bar'), 'get')
    self.assertEqual(30, proxy.get('foo', 32), 'get')
    self.assertEqual(32, proxy.get('invalid key', 32), 'get')

    self.assertEqual(['foo', 'bar'], list(iter(proxy)), 'iter')
    self.assertEqual(['foo', 'bar'], list(proxy.keys()), 'keys')
    self.assertEqual([30, 31], list(proxy.values()), 'values')

    self.assertEqual((('foo', 30), ('bar', 31)), tuple(proxy.items()), 'items')

    del proxy['foo']

    try:
      del proxy['invalid key']
      self.fail("No exception for bad key.")
    except KeyError:
      pass

    self.assertEqual({'bar': 31}, proxy, 'delitem')
    self.assertDbDictPosition(srcList, 'delitem')

    self.assertEqual(True, 'bar' in proxy, 'in')
    self.assertEqual(False, 'foo' in proxy, 'in')

    proxy.clear()
    self.assertEqual({}, proxy, 'clear')

    proxy['foo'] = 30
    proxy['bar'] = 31

    value = proxy.setdefault('bar', 32)
    self.assertEqual(31, value, 'setdefault no change')
    self.assertEqual({'foo': 30, 'bar': 31}, proxy, 'setdefault')
    self.assertDbDictPosition(srcList, 'setdefault')

    value = proxy.setdefault('baz', 32)
    self.assertEqual(32, value, 'setdefault change')
    self.assertEqual({'foo': 30, 'bar': 31, 'baz': 32}, proxy, 'setdefault')
    self.assertDbDictPosition(srcList, 'setdefault')

    value = proxy.pop('invalid key', 33)

    self.assertEqual(33, value, 'pop no match')
    self.assertEqual({'foo': 30, 'bar': 31, 'baz': 32}, proxy, 'pop no match')
    self.assertDbDictPosition(srcList, 'pop')

    try:
      value = proxy.pop('invalid key')
      self.fail("No exception for no key and no default.")
    except KeyError:
      pass

    self.assertEqual({'foo': 30, 'bar': 31, 'baz': 32}, proxy, 'pop no match')

    value1 = proxy.pop('foo', 33)

    self.assertEqual(30, value1, 'pop match')
    self.assertEqual({'bar': 31, 'baz': 32}, proxy, 'pop match')
    self.assertDbDictPosition(srcList, 'pop')

    value2 = proxy.popitem()

    self.assertEqual(('baz', 32), value2, 'popitem')
    self.assertEqual({'bar': 31}, proxy, 'popitem')
    self.assertDbDictPosition(srcList, 'popitem')

    proxy.update({'foo': 34, 'bar': 35}, baz=36)
    self.assertEqual({'foo': 34, 'bar': 35, 'baz': 36}, proxy, 'update dict')
    self.assertDbDictPosition(srcList, 'update')

    proxy.update((('foo', 37), ('bar', 38)), baz=39)
    self.assertEqual({'foo': 37, 'bar': 38, 'baz': 39}, proxy, 'update list')
    self.assertDbDictPosition(srcList, 'update list')

    try:
      proxy.update((('foo', 37, 'too many items'),))
      self.fail("No exception for too many items in the tuple.")
    except ValueError:
      pass

    try:
      hash(proxy)
      self.fail("No exception for hash.")
    except TypeError:
      pass

    self.assertIsNotNone(str(proxy), 'str')
    self.assertIsNotNone(proxy.dumpAll(), 'dumpAll')

    proxy.clear()
    try:
      proxy.popitem()
      self.fail("No exception for popitem empty.")
    except KeyError:
      pass

    try:
      proxy.update({}, {})
      self.fail("No exception for update with multiple args.")
    except TypeError:
      pass

    self.assertEqual(True, proxy != {'a': 'b'}, 'ne')

  #-----------------------------------------------------------------
  def assertDbDictPosition(self, srcList: List[Any], msg: str) -> None:
    '''Checks that dbDictPosition is correct.

    Each item in the list must have a dbDictPostition, and it must
    match its index in the list.
    '''

    msg += ": dict position"

    for i, item in enumerate(srcList):
      self.assertEqual(i, item.dbDictPosition, msg)
