#-------------------------------------------------------------------
#  DbListProxy.py
#
#  The DbListProxy class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the DbListProxy class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import ReflectionUtil
from collections.abc import MutableSequence as AbcMutableSequence
from typing import Any
from typing import Generic
from typing import Iterator
from typing import Iterable
from typing import List
from typing import overload
from typing import Type
from typing import TYPE_CHECKING
from typing import TypeVar
from typing import Union
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
# The type of the item that is stored in the list.
DbListItem = TypeVar('DbListItem')

#===================================================================
# The wrapper class that holds each list item.
# Since this class is created at runtime, we have to use 'Any'.
# pylint: disable=invalid-name
DbListItemObject = Any
# pylint: enable=invalid-name

#===================================================================
# Hack to get around bug: https://github.com/python/mypy/issues/5264
if TYPE_CHECKING:

  # pylint: disable=invalid-name
  MutableSequenceBase = AbcMutableSequence
  # pylint: enable=invalid-name

else:

  class MutableSequenceMaker:
    def __getitem__(self, *args):
      return AbcMutableSequence

  # pylint: disable=invalid-name
  MutableSequenceBase = MutableSequenceMaker()
  # pylint: enable=invalid-name

#===================================================================
class DbListProxy(MutableSequenceBase[DbListItem],
                  Generic[DbListItem]):
  '''A class that acts like a list, but is backed by a database object list.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               itemClass: Type[DbListItemObject],
               itemAttrName: str,
               srcList: List[DbListItemObject]) -> None:
    '''Creates a new DbListProxy.
    '''

    # The list of objects that this list is proxying.
    self.srcList: List[DbListItemObject] = srcList

    # The class to use to create new objects.
    self.itemObjectClass: Type[DbListItemObject] = itemClass

    # The name of the data attribute on the itemObjectClass.
    self.itemAttrName: str = itemAttrName

  #-----------------------------------------------------------------
  def updateDbListPosition(self) -> None:
    '''Updates each dbListPostition to match its current possition in the list.
    '''

    for i, item in enumerate(self.srcList):
      item.dbListPosition = i # type: ignore

  #-----------------------------------------------------------------
  def __len__(self) -> int:
    return len(self.srcList)

  # pylint: disable=function-redefined

  #-----------------------------------------------------------------
  @overload
  def __getitem__(self, index: int) -> DbListItem:
    pass

  #-----------------------------------------------------------------
  @overload
  def __getitem__(self, index: slice) -> List[DbListItem]:
    pass

  #-----------------------------------------------------------------
  def __getitem__(self, index: Union[int, slice]) -> Any:
    # The return type of this function should be:
    #
    #   Union[DbListItem, List[DbListItem]]
    #
    # but there is a bug in MyPy 0.630 where mypy reports:
    #
    #   "Overloaded function implementation cannot produce
    #    return type of signature 2"
    #
    # Using "Any" is the workaround.

    # Single-index get.
    if not isinstance(index, slice):
      return getattr(self.srcList[index], self.itemAttrName)

    # Range get.

    ret = []

    objs = self.srcList.__getitem__(index)
    for obj in objs:
      ret.append(getattr(obj, self.itemAttrName))

    return ret

  #-----------------------------------------------------------------
  @overload
  def __setitem__(self, index: int, value: DbListItem) -> None:
    pass

  #-----------------------------------------------------------------
  @overload
  def __setitem__(self, index: slice, value: Iterable[DbListItem]) -> None:
    pass

  #-----------------------------------------------------------------
  def __setitem__(self,
                  index: Union[int, slice],
                  value: Any) -> None:
    # The type of 'value' should be:
    #
    #   Union[DbListItem, List[DbListItem]]
    #
    # but there is a bug in MyPy 0.630 where mypy reports:
    #
    #   "Overloaded function implementation does not accept
    #    all possible arguments of signature 2"
    #
    # Using "Any" is the workaround.
    # Single-index assign.

    if not isinstance(index, slice):
      setattr(self.srcList[index], self.itemAttrName, value)
      return

    # Assign to a range.

    assert isinstance(value, list)

    if index.stop is None:
      stop = len(self)
    elif index.stop < 0:
      stop = len(self) + index.stop
    else:
      stop = index.stop

    step = index.step or 1
    start = index.start or 0

    sliceIndices = range(start, stop, step)
    lenIndices = len(sliceIndices)

    # If a step is set, the assigned list must be the exact size of
    # the slice.

    if index.step not in (None, 1) and len(value) != lenIndices:
      msg = ("Attempting to assign sequence of size %s to "
             "extended slice of size %s." % (len(value), lenIndices))
      raise ValueError(msg)

    # If the slice is bigger than the assigned list, shrink to fit.
    if lenIndices > len(value):
      delBegin = start + len(value)
      del self.srcList[delBegin:stop]

    # If the assigned list is bigger than the slice, grow to fit.
    if len(value) > lenIndices:
      extraValues = value[lenIndices:]
      for i, extraValue in enumerate(extraValues):
        self.insert(stop + i, extraValue)

    # Copy the assigned list to the same-size part of the slice.
    for theIndex, theValue in zip(sliceIndices, value):
      setattr(self.srcList[theIndex], self.itemAttrName, theValue)

  # pylint: enable=function-redefined

  #-----------------------------------------------------------------
  def __delitem__(self, index: Union[int, slice]) -> None:

    del self.srcList[index]
    self.updateDbListPosition()

  #-----------------------------------------------------------------
  def __contains__(self, value: object) -> bool:

    for obj in self.srcList:
      if getattr(obj, self.itemAttrName) == value:
        return True
    return False

  #-----------------------------------------------------------------
  def __iter__(self) -> Iterator[DbListItem]:
    """Iterate over proxied values.

    For the actual domain objects, iterate over .srcList instead or
    just use the underlying collection directly from its property
    on the parent.
    """

    for obj in self.srcList:
      yield getattr(obj, self.itemAttrName)

  #-----------------------------------------------------------------
  def append(self, value: DbListItem) -> None:

    obj = self.itemObjectClass()
    setattr(obj, self.itemAttrName, value)
    obj.dbListPosition = len(self.srcList) # type: ignore

    self.srcList.append(obj)

  #-----------------------------------------------------------------
  def count(self, value: DbListItem) -> int:

    return list(self).count(value)

  #-----------------------------------------------------------------
  def extend(self, values: Iterable[DbListItem]) -> None:

    for value in values:
      self.append(value)

  #-----------------------------------------------------------------
  def insert(self, index: int, value: DbListItem) -> None:

    obj = self.itemObjectClass()
    setattr(obj, self.itemAttrName, value)
    obj.dbListPosition = index # type: ignore

    self.srcList.insert(index, obj)

    self.updateDbListPosition()

  #-----------------------------------------------------------------
  def pop(self, index: int = -1) -> DbListItem:

    obj = self.srcList.pop(index)
    self.updateDbListPosition()

    return getattr(obj, self.itemAttrName)

  #-----------------------------------------------------------------
  def remove(self, value: DbListItem) -> None:

    for i, val in enumerate(self):

      if val == value:
        del self.srcList[i]
        self.updateDbListPosition()
        return

    raise ValueError("value not in list")

  #-----------------------------------------------------------------
  def reverse(self) -> None:
    """Not supported, use reversed(mylist)"""

    raise TypeError("Not supported, use reversed(mylist)")

  #-----------------------------------------------------------------
  def sort(self, *args, **kwargs) -> None:
    """Not supported, use sorted(mylist)"""

    raise TypeError("Not supported, use sorted(mylist)")

  #-----------------------------------------------------------------
  def clear(self) -> None:

    del self.srcList[0:len(self.srcList)]

  #-----------------------------------------------------------------
  def __eq__(self, other) -> bool:
    return list(self) == other

  #-----------------------------------------------------------------
  def __ne__(self, other) -> bool:
    return list(self) != other

  #-----------------------------------------------------------------
  def __lt__(self, other) -> bool:
    return list(self) < other

  #-----------------------------------------------------------------
  def __le__(self, other) -> bool:
    return list(self) <= other

  #-----------------------------------------------------------------
  def __gt__(self, other) -> bool:
    return list(self) > other

  #-----------------------------------------------------------------
  def __ge__(self, other) -> bool:
    return list(self) >= other

  #-----------------------------------------------------------------
  def __add__(self, iterable: Iterable[DbListItem]) -> List[DbListItem]:
    try:
      other = list(iterable)
    except TypeError:
      return NotImplemented
    return list(self) + other

  #-----------------------------------------------------------------
  def __radd__(self, iterable: Iterable[DbListItem]) -> List[DbListItem]:
    try:
      other = list(iterable)
    except TypeError:
      return NotImplemented
    return other + list(self)

  #-----------------------------------------------------------------
  def __mul__(self, n: int) -> List[DbListItem]:
    if not isinstance(n, int):
      return NotImplemented
    return list(self) * n
  __rmul__ = __mul__

  #-----------------------------------------------------------------
  def __iadd__(self, iterable: Iterable[DbListItem]) -> 'DbListProxy':
    self.extend(iterable)
    return self

  #-----------------------------------------------------------------
  def __imul__(self, n: int) -> 'DbListProxy':
    # unlike a regular list *=, proxied __imul__ will generate unique
    # backing objects for each copy.  *= on proxied lists is a bit of
    # a stretch anyhow, and this interpretation of the __imul__ contract
    # is more plausibly useful than copying the backing objects.
    if not isinstance(n, int):
      return NotImplemented
    if n == 0:
      self.clear()
    elif n > 1:
      self.extend(list(self) * (n - 1))
    return self

  #-----------------------------------------------------------------
  def copy(self) -> List[DbListItem]:
    return list(self)

  #-----------------------------------------------------------------
  def __repr__(self) -> str:
    return repr(list(self))

  #-----------------------------------------------------------------
  def __hash__(self) -> int:
    raise TypeError("%s objects are unhashable" % type(self).__name__)

  #-----------------------------------------------------------------
  # Add the list docstrings to DbListProxy's methods.
  ReflectionUtil.copyDocstrings(list, locals())
