#-------------------------------------------------------------------
#  DbDict.py
#
#  The DbDict class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the DbDict class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .Column import Column
from .Column import manyToOne
from .Column import PrimaryKeyColumn
from .DbDictProxy import DbDictProxy
from collections import OrderedDict
from ai.axe.util import StringUtil
from sqlalchemy.orm.relationships import RelationshipProperty
from typing import Any
from typing import Callable
from typing import Dict
from typing import Generic
from typing import List
from typing import Optional
from typing import overload
from typing import Tuple
from typing import Type
from typing import TypeVar
from typing import Union
import ai.axe.db.alchemy.orm
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
# The type of a key that is stored in the dict.
DbDictKey = TypeVar('DbDictKey')

#===================================================================
# The type of a value that is stored in the dict.
DbDictValue = TypeVar('DbDictValue')

#===================================================================
# The wrapper class that holds each dict item, and is stored in the DB.
DbDictItemObject = TypeVar('DbDictItemObject')

#===================================================================
# The object that owns this dict.  (This dict is a field of the object.)
DbDictOwner = TypeVar('DbDictOwner')

#===================================================================
# We can't use the sqlalchemy.Column type for our column objects
# in this class, because we're using mypy stub trickery to
# make Columns work as DB object class/instance members with different types.
# Until someone figures out a better way, we have to use the 'Any'
# type for those column objects.  We use this columnAny alias
# to at least make clear which variables are really Column objects.
columnAny = Any

#===================================================================
class DbDict(Generic[DbDictKey, DbDictValue]):
  '''A dict attribute that is managed by SQL Alchemy.

  You can create a data attribute like so:

    class Foo(SqlBase):

      tools = DbDict(str, int)

  Now tools is a dict of strings:int (in Python) that gets saved to
  a table whose schema is generated and managed for you.

  By default the table has a name that is your class name plus the
  dict attribute name, and a data column, and a foreign key column.
  For example, the above mapping code would result in this child table:

    table foo_tools (
      id serial,            -- Primary key.
      foo_id int,           -- Foreign key to Foo table.
      db_dict_position int, -- The index in the dict.
      keys varchar          -- The dict key.
      values int            -- The dict value.
    );

  To customize the behavior of the dict, there are several optional
  arguments you can provide.  You can specify a Column
  instead of just a type:

      tools = DbDict(Column("tool",
                            String,
                            values=('fertility', 'water')),
                     Integer)

  You can specify the column to sort the dict by:

      tools = DbDict(String,
                     Integer,
                     orderBy="key")

  By default, the dict items are sorted by the order in which they were
  inserted into the dict (like OrderedDict).

  You can specify the tableArgs to be added to the generated child table.

      tools = DbDict(str,
                     int,
                     tableArgs=(UniqueConstraint('cropZoneId', 'key'),
                                UniqueConstraint('key', 'value'),))

  You can specify the table name you'd like the generated child table to have:

      tools = DbDict(str, int, tableName="foobar_tools")
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               keyType: Type[DbDictKey],
               valueType: Type[DbDictValue],
               keyColumn: Optional[columnAny] = None,
               valueColumn: Optional[columnAny] = None,
               tableName: Optional[str] = None,
               joinColumn: str = 'id',
               tableArgs: Tuple[Any, ...] = (),
               orderBy: str = 'dbDictPosition') -> None:
    '''Creates a new DbDict.

    @param keyType A Python type (such as str, int, float, bool) that
                   can be a single DB column.
    @param valueType A Python type (such as str, int, float, bool) that
                     can be a single DB column.
    @param keyColumn A Column object.  If not provided, will be created
                     from the keyType
    @param valueColumn A Column object.  If not provided, will be created
                       from the valueType
    '''

    # The Column object to create on the child table for the key data.
    self.keyColumn: columnAny

    # The Column object to create on the child table for the value data.
    self.valueColumn: columnAny

    if keyColumn is None:

      # Create a Column from a type.
      self.keyColumn = ai.axe.db.alchemy.Column(keyType)

    else:

      self.keyColumn = keyColumn

    if valueColumn is None:

      # Create a Column from a type.
      self.valueColumn = ai.axe.db.alchemy.Column(valueType)

    else:

      self.valueColumn = valueColumn

    # The name of the child table to create.
    #
    # If not provided, a name will be auto-generated.
    self.tableName: Optional[str] = tableName

    # Name of the parent table column to join to.  Defaults to 'id'.
    self.joinColumn: str = joinColumn

    # Any __table_args__ to be applied to the child class.
    self.tableArgs: Tuple[Any, ...] = tableArgs

    # An order by constraint to use when loading.
    self.orderBy: str = orderBy

    # The generated child object class that is the child table.
    self.childClass: Optional[type] = None

    # The relationship from the parent to the child collection.
    self.relationshipAttrName: Optional[str] = None

    # The relationship from the parent to the child collection.
    self.relationship: Optional[RelationshipProperty] = None

  #-----------------------------------------------------------------
  def containsKey(self, target: Any) -> Any:
    '''Used in a SQL query to test whether this dict contains a key.

    @param target The target value to be tested for.
    '''

    # Forward the query to the child class attribute.
    return getattr(self.childClass, "key").contains(target)

  #-----------------------------------------------------------------
  def containsValue(self, target: Any) -> Any:
    '''Used in a SQL query to test whether this dict contains a value.

    @param target The target value to be tested for.
    '''

    # Forward the query to the child class attribute.
    return getattr(self.childClass, "value").contains(target)

  #-----------------------------------------------------------------
  def process(self,
              parentClassName: str,
              baseClasses: List[type],
              parentAttrs: Dict[str, Any],
              attrName: str) -> None:
    '''Processes the specified class, setting up dict attributes.

    @param parentClassName The class that has the DbDict as a member.
    @param baseClasses Base classes of the parent class.
    @param parentAttrs The map of attributes of the parent classs.
    @param attrName The name of the attribute that this DbDict is assigned to.
    '''

    # Set up the table name.

    if self.tableName is not None:
      childTableName = self.tableName
    else:
      childTableName = (StringUtil.camelCaseToUnderscores(parentClassName) +
                        '_' +
                        StringUtil.camelCaseToUnderscores(attrName))
    childClassName = parentClassName + StringUtil.capitalize(attrName)

    ######################################
    # Create the child table class.

    attrs: Dict[str, Any] = OrderedDict()

    # The DB table name.
    attrs['__tablename__'] = childTableName

    # The Database ID column.
    attrs['id'] = PrimaryKeyColumn()

    # The dict position column.
    attrs['dbDictPosition'] = Column(int, index=True)

    # The key column.
    attrs["key"] = self.keyColumn

    # The value column.
    attrs["value"] = self.valueColumn

    # The foreign key column.
    fkAttrName = StringUtil.initialLower(parentClassName + "Id")
    joinColumn = parentAttrs['__tablename__'] + "." +  self.joinColumn
    attrs[fkAttrName] = manyToOne(joinColumn)

    # Add a __repr__  method.
    attrs['__repr__'] = self.createRepr()

    # Table args.
    if self.tableArgs:
      attrs['__table_args__'] = self.tableArgs

    assert len(baseClasses) == 1, baseClasses

    childClass = type(str(childClassName), tuple(baseClasses), attrs)

    ######################################
    # Modify the parent class.

    # Create the relationship with attribute name attrName + 'Object'.
    relationshipAttrName = attrName + 'Objects'
    kwargs = {}
    if self.orderBy:
      if not hasattr(childClass, self.orderBy):
        msg = "Error creating %s.%s.  " % (parentClassName, attrName)
        msg += "You passed in orderBy='%s', "
        msg += "but there's no static data attribute called %s "
        msg += "on the list table class."
        msg = msg % (self.orderBy, self.orderBy)
        raise ValueError(msg)
      kwargs['order_by'] = getattr(childClass, self.orderBy)

    relationship = ai.axe.db.alchemy.orm.relationship
    parentAttrs[relationshipAttrName] = relationship(childClassName, **kwargs)

    # Make the child class available to users.
    parentAttrs[attrName + 'Class'] = childClass

    self.childClass = childClass
    self.relationshipAttrName = relationshipAttrName
    self.relationship = parentAttrs[relationshipAttrName]

  #-----------------------------------------------------------------
  def createObject(self,
                   key:
                   DbDictKey,
                   value: DbDictValue) -> DbDictItemObject:
    '''Returns a new list object containing the specified list value.

    @param value The list value to create an object for.

    @return A new object of type self.childClass.
    '''

    assert self.childClass is not None

    obj = self.childClass()
    setattr(obj, "key", key)
    setattr(obj, "value", value)
    return obj

  #-----------------------------------------------------------------
  def createRepr(self) -> Callable[[Any], str]:
    '''Returns a __repr__ method for an object with a single attribute.
    '''

    def newReprMethod(self) -> str:
      '''Formats this object into a string.
      '''

      attrs = ["key", "value"]
      return StringUtil.formatRepr(self, attrs)

    return newReprMethod

  # pylint: disable=function-redefined

  #-----------------------------------------------------------------
  @overload
  def __get__(self,
              obj: None,
              objType: Type[DbDictOwner]) -> 'DbDict':
    pass

  #-----------------------------------------------------------------
  @overload
  def __get__(self,
              obj: DbDictOwner,
              objType: Type[DbDictOwner]) -> DbDictProxy[DbDictKey,
                                                         DbDictValue]:
    pass

  #-----------------------------------------------------------------
  def __get__(self,
              obj: Union[None, DbDictOwner],
              objType: Type[DbDictOwner]) -> Union['DbDict',
                                                   DbDictProxy[DbDictKey,
                                                               DbDictValue]]:
    '''Descriptor get method.

    This method is called in two places.  Assume there's a class Foo
    with a DbDict called bar.

      class Foo:
        bar = DbDict(String)

      foo = Foo()

      values = foo.bar   # Case 1: object
      relationship = Foo.bar  # Case 2: class

    In case 1, obj will be 'foo' and objType will be 'Foo'.
    In case 2, obj will be None, and objType will be 'Foo'.
    '''

    if not obj:

      # This is being called on the class, so just return myself.
      # This allows to user to call 'contains' on me in queries.

      return self

    else:

      assert self.relationshipAttrName is not None
      assert self.childClass is not None

      # This is being called on an object, so return a list proxy.

      objectList = getattr(obj, self.relationshipAttrName)
      return DbDictProxy[DbDictKey,
                         DbDictValue](self.childClass,
                                      "key",
                                      "value",
                                      objectList)

  # pylint: enable=function-redefined

  #-----------------------------------------------------------------
  def __set__(self,
              obj: DbDictOwner,
              values: Dict[DbDictKey, DbDictValue]) -> None:
    '''Descriptor set method.

    This method is called when the user assigns values to the child
    list.  Assume there's a class Foo with a DbDict called bar.

      class Foo:
        bar = DbDict(int, int)

      foo = Foo()

      foo.bar = {1: 4, 2: 5, 3: 6} # Called here.
    '''

    assert self.relationshipAttrName is not None

    # pylint: disable=C0200

    valueItems: List[Tuple[DbDictKey, DbDictValue]] = list(values.items())

    objectList = getattr(obj, self.relationshipAttrName)

    while len(objectList) > len(values):
      objectList.pop()

    for i in range(len(objectList)):
      key, value = valueItems[i]
      setattr(objectList[i], "key", key)
      setattr(objectList[i], "value", value)

    for i in range(len(objectList), len(values)):
      key, value = valueItems[i]
      objectList.append(self.createObject(key, value))

    # Update the dbDictPosition values to be correct.
    for i, item in enumerate(objectList):
      item.dbDictPosition = i

  #-----------------------------------------------------------------
  def __delete__(self, obj: DbDictOwner) -> None:
    '''This method is called when the user deletes the list.

      class Foo:
        bar = DbDict(String)

      foo = Foo()

      del foo.bar # Called here.
    '''

    assert self.childClass is not None

    name = self.childClass.__name__
    msg = "You are not allowed to delete the list " + name + "."
    raise NotImplementedError(msg)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['column', 'tableName', 'joinColumn', 'tableArgs']

    return StringUtil.formatRepr(self, attrs)
