#-------------------------------------------------------------------
#  Column.py
#
#  The Column module.
#
#  Copyright 2013 Applied Invention, LLC.
#-------------------------------------------------------------------

'''A Column constructor function that overrides the sqlalchemy.Column ctor.

Columns are non-nullable by default.

If a min and/or max value is supplied, a check constraint will be generated.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import AxeMapping
from collections import OrderedDict
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.sql.type_api import TypeEngine
from typing import Any
from typing import Dict
from typing import List
from typing import Optional
from typing import TYPE_CHECKING
from typing import Tuple
from typing import Type
from typing import TypeVar
from typing import Union
import sqlalchemy
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
typeMap: Dict[type, TypeEngine] = OrderedDict()

#-------------------------------------------------------------------
# Supress 'unused Optional' error (Optional is only used in type strings).
unused = Optional

#-------------------------------------------------------------------
def addTypeMapping(pythonType: type, dbType: TypeEngine) -> None:
  '''Adds a type mapping to be converted on the fly.

   Allows the user to specify a DB type by using a simpler type.

  For example, to specify an Integer type, you do:

    from sqlalchemy import Integer
    numWheels = Column(Integer)

  But if you first do:

    from sqlalchemy import Integer
    Column.addTypeMapping(int, Integer)

  then the user can do the simpler:

    numWheels = Column(int)
  '''

  typeMap[pythonType] = dbType

#-------------------------------------------------------------------
# Generic type for the member variables that are mapped to a column.
PythonType = TypeVar('PythonType')

#-------------------------------------------------------------------
def createPrimaryKeyColumn(*args,
                           **kwargs) -> 'sqlalchemy.Column[Optional[int]]':

  '''Creates a new primary key Column object.

  See the createColumn() function for details of parameters.

  Note that the type is Optional[] even though nullable=False,
  because a transient object's self.id is None until it is persisted.
  '''

  kwargs.setdefault('primary_key', True)

  return createColumn(int, *args, **kwargs)

#-------------------------------------------------------------------
def createUuidPrimaryKeyColumn(*args,
                               **kwargs) -> 'sqlalchemy.Column[Optional[str]]':

  '''A primary key Column that uses UUIDs rather than auto-increment ints.

  See the createColumn() function for details of parameters.

  Note that the type is Optional[] even though nullable=False,
  because a transient object's self.id is None until it is persisted.
  '''

  kwargs.setdefault('primary_key', True)

  return createColumn(str, UUID, *args, **kwargs)

#-------------------------------------------------------------------
def createNullableColumn(pythonType: Type[PythonType],
                         *args,
                         **kwargs) -> 'sqlalchemy.Column[Optional[PythonType]]':
  '''Creates a new nullable Column object.

  See the createColumn() function for details of parameters.
  '''

  # The reason for this function is that mypy does not allow Optional[]
  # or Union[] to be used in a generic function.
  #
  # So this would work:
  #
  #    column = sqlalchemy.Column[Optional[date]]();
  #
  # The type of 'column' is correctly sqlalchemy.Column[Optional[date]].
  #
  # But this does not work:
  #
  #    column = createColumn(Optional[date])
  #
  # MyPy considers the type of 'column' to be 'sqlalchemy.Column[Any]'.
  #
  # Since createColumn(T) -> Column[T], doesn't work when T is Optional[],
  # we had to create this 'createNullableColumn(T) -> Column[Optional[T]].

  # The Column is non-nullable by default.
  kwargs.setdefault('nullable', True)

  return createColumn(pythonType, *args, **kwargs)

#-------------------------------------------------------------------
def createColumn(pythonType: Type[PythonType],
                 *args,
                 **kwargs) -> 'sqlalchemy.Column[PythonType]':
  '''Creates a new Column object.

  The first argument must be the Python type of the column:
  a Python type object such as bool, int, float, date, datetime, or str.

    Column(datetime)

  To create a NULLABLE column, use the NullableColumn.

    from ai.axe.db.alchemy import NullableColumn

    NullableColumn(datetime)

  Optionally, a SQLAlchemy type object (such as String or Integer) may
  be passed as the second argument.  If no such object is passed in,
  the SQLAlchemy type will be deduced from the Python type.

  In practice, a SQLALchemy type will almost never be needed.  One reason
  to pass one in is to specify the length of a VARCHAR column.
  For example, to create a VARCHAR(25) column:

    Column(str, String(256))

  All other arguments are passed through to the sqlalchemy.Column constructor.

  The name of the column will be the name of the data attribute this
  Column is assigned to.  For example, this will create a column called 'foo':

    foo = Column(str)

  If you want to explicitly set the name of the column, use the 'name'
  keyword argument.

    foo = Column(str, name='my_column_name')

  @return A new sqlalchemy.Column object.
  '''

  info: Dict[str, Any] = {}

  if isSqlAlchemyType(pythonType):
    msg = 'You forgot to pass a python type as the first argument to the '
    msg += 'Column() constructor.  You passed in a SQLAlchemy type: %s'
    msg = msg % pythonType
    raise TypeError(msg)

  # If the user specified a SQLAlchemy type, use that.
  if len(args) >= 1 and isSqlAlchemyType(args[0]):
    sqlAlchemyType = args[0]
    args = args[1:]

  else:
    # Convert the passed-in python type to a SQL Alchemy type.
    if pythonType not in typeMap:
      validStr = ', '.join([str(x) for x in typeMap])
      msg = 'Invalid Column(pythonType) argument.\n'
      msg += 'You passed in: %s \n' % pythonType
      msg += 'Valid types: %s' % validStr
    sqlAlchemyType = typeMap[pythonType]

  # Columns are non-nullable by default.
  kwargs.setdefault('nullable', False)

  # Columns have an index by default.
  #
  # Primary keys are always indexed, so don't add an index to them.
  #
  if not kwargs.get('primary_key', False):
    kwargs.setdefault('index', True)

  # If a min and/or max value is supplied, a check constraint will be generated.
  for attrName in AxeMapping.columnAttrs:
    if attrName in kwargs:
      info[attrName] = kwargs.pop(attrName)

  if info:
    kwargs['info'] = info

  # Our mypy stubs have Column as a Generic[PythonType], but the real
  # Column isn't generic, so need to pick the correct call depending
  # on whether we're type-checking or running.
  # Also, we really want this type to be Column[PythonType], but that
  # doesn't seem to be allowed (yet?), so we have to use Column[Any].
  if TYPE_CHECKING:
    # pylint: disable=unsubscriptable-object
    columnCtor = sqlalchemy.Column[Any]
    # pylint: enable=unsubscriptable-object
  else:
    columnCtor = sqlalchemy.Column

  return columnCtor(sqlAlchemyType, *args, **kwargs)

#-------------------------------------------------------------------
# This can be used as a replacement for the SQLAlchemy Column() constructor.
# pylint: disable = C0103
Column = createColumn
NullableColumn = createNullableColumn
PrimaryKeyColumn = createPrimaryKeyColumn
UuidPrimaryKeyColumn = createUuidPrimaryKeyColumn
# pylint: enable = C0103

#-------------------------------------------------------------------
def oneToOne(foreignColumn: Union[str, sqlalchemy.ForeignKey],
             *args,
             **kwargs) -> 'sqlalchemy.Column[int]':
  '''Creates a new Column object for a one-to-one.

  A one-to-one is like a many-to-one, but with a unique constraint
  on the foreign key column.

  This code:

    fieldId = oneToOne('fields.id')

  is equivalent to:

    fieldId = Column(ForeignKey('fields.id',
                                ondelete='cascade'),
                     nullable=False,
                     index=True,
                     unique=True)

  @param sqlType A SQLAlchemy type object, such as sqlalchemy.Integer.
  @param foreignColumn Can be the string name of a foreign column,
                       or a ForeignKey object.  If this is a a ForeignKey
                       object it is used as is.  If it is a string,
                       a ForeignKey object will be built for it automatically.

  @return A new Column object.
  '''

  kwargs.setdefault('unique', True)

  return manyToOne(foreignColumn, *args, **kwargs)

#-------------------------------------------------------------------
def nullableOneToOne(foreignColumn: Union[str, sqlalchemy.ForeignKey],
                     *args,
                     **kwargs) -> 'sqlalchemy.Column[Optional[int]]':
  '''Creates a new Column object for a one-to-one.

  A one-to-one is like a many-to-one, but with a unique constraint
  on the foreign key column.

  This code:

    fieldId = nullableOneToOne('fields.id')

  is equivalent to:

    fieldId = Column(ForeignKey('fields.id',
                                ondelete='set null'),
                     nullable=True,
                     index=True,
                     unique=True)

  @param sqlType A SQLAlchemy type object, such as sqlalchemy.Integer.
  @param foreignColumn Can be the string name of a foreign column,
                       or a ForeignKey object.  If this is a a ForeignKey
                       object it is used as is.  If it is a string,
                       a ForeignKey object will be built for it automatically.

  @return A new Column object.
  '''

  if isinstance(foreignColumn, sqlalchemy.ForeignKey):
    foreignKey = foreignColumn
  else:
    foreignKey = sqlalchemy.ForeignKey(foreignColumn, ondelete='set null')

  kwargs.setdefault('nullable', True)

  return oneToOne(foreignKey, *args, **kwargs) # type: ignore

#-------------------------------------------------------------------
def manyToOne(foreignColumn: Union[str, sqlalchemy.ForeignKey],
              *args,
              **kwargs) -> 'sqlalchemy.Column[int]':
  '''Creates a new Column object for a many-to-one.

  This code:

    fieldId = manyToOne('fields.id')

  is equivalent to:

    fieldId = Column(ForeignKey('fields.id',
                                ondelete='cascade'),
                     nullable=False,
                     index=True)

  @param foreignColumn Can be the string name of a foreign column,
                       or a ForeignKey object.  If this is a a ForeignKey
                       object it is used as is.  If it is a string,
                       a ForeignKey object will be built for it automatically.

  @return A new Column object.
  '''

  if isinstance(foreignColumn, sqlalchemy.ForeignKey):
    foreignKey = foreignColumn
  else:
    foreignKey = sqlalchemy.ForeignKey(foreignColumn, ondelete='cascade')

  kwargs.setdefault('nullable', False)
  kwargs.setdefault('index', True)

  column = sqlalchemy.Column(foreignKey, *args, **kwargs)

  return column

#-------------------------------------------------------------------
def nullableManyToOne(foreignColumn: Union[str, sqlalchemy.ForeignKey],
                      *args,
                      **kwargs) -> 'sqlalchemy.Column[Optional[int]]':
  '''Creates a new Column object for a many-to-one.

  This code:

    fieldId = manyToOne('fields.id')

  is equivalent to:

    fieldId = Column(ForeignKey('fields.id',
                                ondelete='set null'),
                     nullable=True,
                     index=True)

  @param foreignColumn Can be the string name of a foreign column,
                       or a ForeignKey object.  If this is a a ForeignKey
                       object it is used as is.  If it is a string,
                       a ForeignKey object will be built for it automatically.

  @return A new Column object.
  '''

  if isinstance(foreignColumn, sqlalchemy.ForeignKey):
    foreignKey = foreignColumn
  else:
    foreignKey = sqlalchemy.ForeignKey(foreignColumn, ondelete='set null')

  kwargs.setdefault('nullable', True)
  kwargs.setdefault('index', True)

  column = sqlalchemy.Column(foreignKey, *args, **kwargs)

  return column

#-------------------------------------------------------------------
def isSqlAlchemyType(value: Any) -> bool:
  '''Returns True if the value is a SQL Alchemy type, such as Integer.
  '''

  # Check whether the value is an object like String(256).

  if isinstance(value, TypeEngine):
    return True

  # Check whether the value is a class like Integer.

  # Note that issubclass() throws an exception if you pass in a non-class,
  # so treat any exception as if issubclass() had returned False.

  try:
    if issubclass(value, TypeEngine):
      return True
  except TypeError:
    pass

  # Anything else is not a SQL Alchemy type.

  return False

#-------------------------------------------------------------------
def optionalType(value: type) -> Tuple[bool, type]:
  '''Private helper.  Determines whether the value is an Optional[].

  @return A (isOptional, baseType) tuple.
  '''

  # pylint: disable=unidiomatic-typecheck

  # Not sure what Union is, but it's not a regular class, so we can't
  # use it with isinstance() or issubclass().  This hack will have to do.

  isUnion = str(type(value)) == 'typing.Union'

  if isUnion:

    # Don't know how to tell mypy that value is a Union, so have to just
    # silence the error.
    args: List[type] = value.__args__ # type: ignore

    if len(args) == 2 and type(None) in args:

      return True, args[0]

  return False, value
