#-------------------------------------------------------------------
#  AlchemySetup.py
#
#  The AlchemySetup module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Setup functions for GMC SQL Alchemy extensions.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .types.AngleType import AngleType
from .types.MultiPolygonType import MultiPolygonType
from .types.PolygonType import PolygonType
from .types.PointType import PointType
from .types.SingleMultiPolygonType import SingleMultiPolygonType
from sqlalchemy import Table
from sqlalchemy import event
from sqlalchemy.engine import Connection
from sqlalchemy.sql.expression import ColumnElement
#
# Import statements go above this line.
#-------------------------------------------------------------------

# pylint: disable = C0103, W0613

#-------------------------------------------------------------------
# All gmc.alchemy.type types.
allGeospatialTypes = (
  MultiPolygonType,
  PointType,
  PolygonType,
  SingleMultiPolygonType
)

#-------------------------------------------------------------------
@event.listens_for(Table, "after_create")
def after_create(table: Table, bind: Connection, **kw):
  '''Called during DDL generation when a table has been created.
  '''

  for column in table.c:

    processGeospatial(table, column, bind)
    processAngle(table, column, bind)

#-------------------------------------------------------------------
def processGeospatial(table: Table,
                      column: ColumnElement,
                      bind: Connection) -> None:

  # Add spatial indices for geospatial columns.
  if isinstance(column.type, allGeospatialTypes):

    sql = 'create index idx_%s_%s on %s using gist (%s)'
    sql = sql % (table.name, column.name, table.name, column.name)

    bind.execute(sql
)

  # Add a check constraint for SingleMultiPolygon type,
  # since a single type can't be specified in the column type.
  if isinstance(column.type, SingleMultiPolygonType):

    sql = ("alter table %s add constraint ck_%s_%s " +
           "check (geometrytype(%s) = 'MULTIPOLYGON'::text or " +
           "geometrytype(%s) = 'POLYGON'::text or %s is null)")
    sql = sql % (table.name, table.name,
                 column.name, column.name, column.name, column.name)

    bind.execute(sql)

#-------------------------------------------------------------------
def processAngle(table: Table,
                 column: ColumnElement,
                 bind: Connection) -> None:

  # Add a check constraint for angles.
  if isinstance(column.type, AngleType):

    bind.execute(AngleType.createConstraintSql(table.name, column.name))
