#-------------------------------------------------------------------
#  DbUtil.py
#
#  The DbUtil module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Database utility functions.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from typing import TextIO
from sqlalchemy.orm.session import Session
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def attrToColumnName(attrName: str) -> str:
  '''Converts a camel-case attribute name to an underscore column name.

  @param attrName The attribute name to convert.

  @return The converted column name.
  '''

  return StringUtil.camelCaseToUnderscores(attrName)

#-------------------------------------------------------------------
def findAndDropAllTables(session: Session,
                         doDrop: bool,
                         statusOut: TextIO) -> None:
  '''Drops all tables found for the current session.

  @param session The DB connection to use.
  @param doDrop Whether to write the commands only, or actually do the drop.
  @param statusOut A file object that status messages will be written to.
  '''

  # Postgres uses schema 'public'.
  # MS-SQL uses schema 'dbo'.

  # We do not drop the spatial_ref_sys table, as this is a bookkeeping
  # table used by postgis.

  # Find the names of tables to drop.

  tableNameSql = '''\
select table_name
from information_schema.tables
where (table_schema = 'public' or
       table_schema = 'dbo') and
      table_name != 'spatial_ref_sys' and
      table_name != 'geography_columns' and
      table_name != 'geometry_columns' and
      table_name != 'raster_columns' and
      table_name != 'raster_overviews'
order by table_name;'''

  resultSet = session.execute(tableNameSql).fetchall()

  tables = [x[0] for x in resultSet]

  # Find the names of constraints to drop.

  constraintNameSql = '''\
select table_name, constraint_name
from information_schema.table_constraints
where (table_schema = 'public' or
       table_schema = 'dbo') and
       table_name != 'spatial_ref_sys' and
       constraint_type = 'FOREIGN KEY'
order by table_name;'''

  resultSet = session.execute(constraintNameSql).fetchall()

  tableConstraints = [(x[0], x[1]) for x in resultSet]

  constraintSql = 'alter table %s drop constraint %s;'

  constraintSqls = []
  for table, constraint in tableConstraints:
    sql = constraintSql % (table, constraint)
    constraintSqls.append(sql)

  tableSqls = [('drop table if exists %s;' % table) for table in tables]

  sqls = constraintSqls + tableSqls

  if doDrop:
    for sql in sqls:
      statusOut.write('  ' + sql + '\n')
      session.execute(sql)
  else:
    for sql in sqls:
      statusOut.write('  ' + sql + '\n')
