#-------------------------------------------------------------------
#  TestClassJsonTypescriptWriter.py
#
#  The TestClassJsonTypescriptWriter module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the ClassJsonTypescriptWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson.jsonTypes.JsonObjRegistry import JsonObjRegistry
from ai.axe.classJson import ClassJsonTypescriptWriter
from ai.axe.build.unittest import AxeSimpleTestCase
from datetime import date
from datetime import datetime
from typing import Dict
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestClassJsonTypescriptWriter(AxeSimpleTestCase):
  '''Unit test for the ClassJsonTypescriptWriter class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister("TestClass1")
    JsonObjRegistry.unregister("TestClass2")
    JsonObjRegistry.unregister("TestClassBase")
    JsonObjRegistry.unregister("TestClassSub1")

  #-----------------------------------------------------------------
  def testWritingBasicTypes(self) -> None:
    '''Test writing a class with primitive types.
    '''

    writer = ClassJsonTypescriptWriter()

    @ClassJsonClass([ClassJsonField('intValue', int)])
    class TestClassInner:

      def __init__(self, intValue) -> None:
        self.intValue = intValue


    @ClassJsonClass([ClassJsonField('boolValue', bool),
                     ClassJsonField('dateValue', date),
                     ClassJsonField('datetimeValue', datetime),
                     ClassJsonField('floatValue', float),
                     ClassJsonField('intValue', int),
                     ClassJsonField('stringValue', str),
                     ClassJsonField('dictValue', {str: int}),
                     ClassJsonField('custClass', TestClassInner),
                     ClassJsonField('boolList', [bool]),
                     ClassJsonField('dateList', [date]),
                     ClassJsonField('datetimeList', [datetime]),
                     ClassJsonField('floatList', [float]),
                     ClassJsonField('intList', [int]),
                     ClassJsonField('stringList', [str]),
                     ClassJsonField('custClassList', [TestClassInner])])
    class TestClass1:

      def __init__(self,
                   boolValue: bool,
                   dateValue: date,
                   datetimeValue: datetime,
                   floatValue: float,
                   intValue: int,
                   stringValue: str,
                   dictValue: Dict[str, int],
                   custClass: TestClassInner,
                   boolList: List[bool],
                   dateList: List[date],
                   datetimeList: List[datetime],
                   floatList: List[float],
                   intList: List[int],
                   stringList: List[str],
                   custClassList: List[TestClassInner]) -> None:

        self.boolValue = boolValue
        self.dateValue = dateValue
        self.datetimeValue = datetimeValue
        self.floatValue = floatValue
        self.intValue = intValue
        self.stringValue = stringValue
        self.dictValue = dictValue
        self.custClass = custClass
        self.boolList = boolList
        self.dateList = dateList
        self.datetimeList = datetimeList
        self.floatList = floatList
        self.intList = intList
        self.stringList = stringList
        self.custClassList = custClassList

    # Test plain text format.

    expected = '''\
export interface TestClass1
{
  boolValue: boolean;
  dateValue: Day;
  datetimeValue: Date;
  floatValue: number;
  intValue: number;
  stringValue: string;
  dictValue: Map<string, number>;
  custClass: TestClassInner;
  boolList: Array<boolean>;
  dateList: Array<Day>;
  datetimeList: Array<Date>;
  floatList: Array<number>;
  intList: Array<number>;
  stringList: Array<string>;
  custClassList: Array<TestClassInner>;
}

class BasicTestClass1
{
  constructor(public boolValue: boolean,
              public dateValue: Day,
              public datetimeValue: Date,
              public floatValue: number,
              public intValue: number,
              public stringValue: string,
              public dictValue: Map<string, number>,
              public custClass: TestClassInner,
              public boolList: Array<boolean>,
              public dateList: Array<Day>,
              public datetimeList: Array<Date>,
              public floatList: Array<number>,
              public intList: Array<number>,
              public stringList: Array<string>,
              public custClassList: Array<TestClassInner>)
  {
  }

  static fromJson(src: TestClass1) : BasicTestClass1
  {
    return new BasicTestClass1(src.boolValue,
                               src.dateValue,
                               src.datetimeValue,
                               src.floatValue,
                               src.intValue,
                               src.stringValue,
                               src.dictValue,
                               src.custClass,
                               src.boolList,
                               src.dateList,
                               src.datetimeList,
                               src.floatList,
                               src.intList,
                               src.stringList,
                               src.custClassList);
  }

  static toJson(src: BasicTestClass1) : TestClass1
  {
    return src;
  }
}
'''

    actual = writer.write(TestClass1, indent=0)

    self.assertEqual(expected, actual, msg="Basic Values")

  #-----------------------------------------------------------------
  def testWritingSubclasses(self) -> None:
    '''Test writing a class with subclasses.
    '''

    writer = ClassJsonTypescriptWriter()

    @ClassJsonClass([], isAbstract=True)
    class TestClassBase:
      def __init__(self) -> None:
        pass

    @ClassJsonClass([ClassJsonField('intValue', int)])
    class TestClassSub1(TestClassBase):

      def __init__(self, intValue: int) -> None:
        TestClassBase.__init__(self)
        self.intValue = intValue

    # Check the subclass.

    expected = 'export interface TestClassSub1 extends TestClassBase\n{'

    actual = writer.write(TestClassSub1, indent=0)

    self.assertEqual(True, actual.startswith(expected), "Subclass")

  #-----------------------------------------------------------------
  def testExtraWhitespceInBaseInterface(self) -> None:
    '''Tests that there's no extra space in a base class interface.
    '''

    # This tests a bug where a class with no properties would
    # generate an extra line in the interface declaration.

    writer = ClassJsonTypescriptWriter()

    # Check the base class.

    expected = '''  export interface TestClassBase
  {
  }
'''

    actual = writer.writeInterface("TestClassBase", [], [], '', 1)

    self.assertEqual(expected, actual, "Base class")

  #-----------------------------------------------------------------
  def testWritingMultiple(self) -> None:
    '''Test writing multiple classes.
    '''

    @ClassJsonClass([ClassJsonField('intValue', int)])
    class TestClass1:

      def __init__(self, intValue: int) -> None:
        self.intValue = intValue

    @ClassJsonClass([ClassJsonField('floatValue', float)])
    class TestClass2:

      def __init__(self, floatValue: float) -> None:
        self.floatValue = floatValue

    jsonObj1 = JsonObjRegistry.getForClass(TestClass1)
    jsonObj2 = JsonObjRegistry.getForClass(TestClass2)

    writer = ClassJsonTypescriptWriter()

    # Test the namespace is correct.

    expectedInterfaces = '''\
//------------------------------------------------------------------
//  classJsonInterfaces.ts
//
// This file contains generated Typescript interfaces for all ClassJson
// classes used in web actions.
//
//------------------------------------------------------------------

// Allow this file to break the 80-char-max-line-length rule.
/* tslint:disable:max-line-length */

// Allow this file to declare members in constructor argument lists.
/* tslint:disable:no-parameter-properties */

import { Day } from '../axe/date/Day';
import { Duration } from '../axe/date/Duration';
import { Map } from '../axe/util/Map';

export interface TestClass1
{
  intValue: number;
}

export interface TestClass2
{
  floatValue: number;
}

'''

    expectedClasses = '''\
//------------------------------------------------------------------
//  classJsonClasses.ts
//
// This file contains private implementations of ClassJson definitions
// used in web actions.
// Do not use these.  Instead use the public interfaces
// in classJsonInterfaces.ts
//
//------------------------------------------------------------------

// Allow this file to break the 80-char-max-line-length rule.
/* tslint:disable:max-line-length */

// Allow this file to declare members in constructor argument lists.
/* tslint:disable:no-parameter-properties */

import { Day } from '../axe/date/Day';
import { Duration } from '../axe/date/Duration';
import { ClassJsonRegistry } from '../axe/classJson/ClassJsonRegistry';
import { ClassJsonDesc } from '../axe/classJson/ClassJsonDesc';
import { JsonAny } from '../axe/classJson/jsonTypes/JsonAny';
import { JsonDate } from '../axe/classJson/jsonTypes/JsonDate';
import { JsonDateTime } from '../axe/classJson/jsonTypes/JsonDateTime';
import { JsonDuration } from '../axe/classJson/jsonTypes/JsonDuration';
import { JsonLink } from '../axe/classJson/jsonTypes/JsonLink';
import { JsonList } from '../axe/classJson/jsonTypes/JsonList';
import { JsonMap } from '../axe/classJson/jsonTypes/JsonMap';
import { JsonObj } from '../axe/classJson/jsonTypes/JsonObj';
import { JsonPrimitiveType } from '../axe/classJson/jsonTypes/JsonPrimitiveType';
import { Map } from '../axe/util/Map';

class BasicTestClass1
{
  constructor(public intValue: number)
  {
  }

  static fromJson(src: TestClass1) : BasicTestClass1
  {
    return new BasicTestClass1(src.intValue);
  }

  static toJson(src: BasicTestClass1) : TestClass1
  {
    return src;
  }
}

class BasicTestClass2
{
  constructor(public floatValue: number)
  {
  }

  static fromJson(src: TestClass2) : BasicTestClass2
  {
    return new BasicTestClass2(src.floatValue);
  }

  static toJson(src: BasicTestClass2) : TestClass2
  {
    return src;
  }
}



export function registerDefaults() : void
{
  ClassJsonRegistry.registry.register('TestClass1',
                                      BasicTestClass1,
                                      false);
  ClassJsonRegistry.registry.register('TestClass2',
                                      BasicTestClass2,
                                      false);
}


export function registerDescs() : void
{
  let desc: ClassJsonDesc = null;

  desc = new ClassJsonDesc("TestClass1");
  desc.addField("intValue", new JsonPrimitiveType("int"));
  ClassJsonRegistry.registry.addDesc(desc);

  desc = new ClassJsonDesc("TestClass2");
  desc.addField("floatValue", new JsonPrimitiveType("float"));
  ClassJsonRegistry.registry.addDesc(desc);
}
'''

    interfaces, classes = writer.writeAll([jsonObj1, jsonObj2], False, indent=0)

    self.assertEqual(expectedInterfaces, interfaces, msg="Multiple interfaces")
    self.assertEqual(expectedClasses, classes, msg="Multiple classes")
