#-------------------------------------------------------------------
#  TestClassJsonSchemaWriter.py
#
#  The TestClassJsonSchemaWriter module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the ClassJsonSchemaWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import date
from datetime import datetime
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson.jsonTypes.JsonObjRegistry import JsonObjRegistry
from ai.axe.classJson import ClassJsonSchemaWriter
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestClassJsonSchemaWriter(AxeSimpleTestCase):
  '''Unit test for the ClassJsonSchemaWriter class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister("TestClass1")
    JsonObjRegistry.unregister("LugNut")
    JsonObjRegistry.unregister("Wheel")
    JsonObjRegistry.unregister("Car")

  #-----------------------------------------------------------------
  def testWritingBasicTypes(self) -> None:
    '''Test writing a schema with primitive types.
    '''

    writer = ClassJsonSchemaWriter()

    @ClassJsonClass([ClassJsonField('boolValue', bool),
                     ClassJsonField('dateValue', date),
                     ClassJsonField('datetimeValue', datetime),
                     ClassJsonField('floatValue', float),
                     ClassJsonField('intValue', int),
                     ClassJsonField('stringValue', str),
                     ClassJsonField('boolList', [bool]),
                     ClassJsonField('dateList', [date]),
                     ClassJsonField('datetimeList', [datetime]),
                     ClassJsonField('floatList', [float]),
                     ClassJsonField('intList', [int]),
                     ClassJsonField('stringList', [str])])
    class TestClass1:

      def __init__(self,
                   boolValue: bool,
                   dateValue: date,
                   datetimeValue: datetime,
                   floatValue: float,
                   intValue: int,
                   stringValue: str,
                   boolList: List[bool],
                   dateList: List[date],
                   datetimeList: List[datetime],
                   floatList: List[float],
                   intList: List[int],
                   stringList: List[str]) -> None:

        self.boolValue = boolValue
        self.dateValue = dateValue
        self.datetimeValue = datetimeValue
        self.floatValue = floatValue
        self.intValue = intValue
        self.stringValue = stringValue
        self.boolList = boolList
        self.dateList = dateList
        self.datetimeList = datetimeList
        self.floatList = floatList
        self.intList = intList
        self.stringList = stringList

    # Test plain text format.

    expected = '''\
{
  "_class" : "TestClass1",
  "boolValue" : <bool>,
  "dateValue" : <date>,
  "datetimeValue" : <datetime>,
  "floatValue" : <float>,
  "intValue" : <int>,
  "stringValue" : <string>,
  "boolList" : <[bool]>,
  "dateList" : <[date]>,
  "datetimeList" : <[datetime]>,
  "floatList" : <[float]>,
  "intList" : <[int]>,
  "stringList" : <[string]>
}'''

    actual = writer.writeClass(TestClass1, indent=0)

    self.assertEqual(expected, actual, msg="Basic Values")

    # Test JSON format.

    expected = '''\
{
  "TestClass1": {
    "boolValue": "bool",
    "dateValue": "date",
    "datetimeValue": "datetime",
    "floatValue": "float",
    "intValue": "int",
    "stringValue": "string",
    "boolList": "[bool]",
    "dateList": "[date]",
    "datetimeList": "[datetime]",
    "floatList": "[float]",
    "intList": "[int]",
    "stringList": "[string]"
  }
}'''

    jsonObjDict = {"TestClass1" : JsonObjRegistry.getForName("TestClass1")}
    actual = writer.writeJsonObjDictToJson(jsonObjDict)

    self.assertEqual(expected, actual, msg="Basic Values JSON")

  #-----------------------------------------------------------------
  def testWritingNested(self) -> None:
    '''Test writing a schema with nested objects.
    '''

    @ClassJsonClass([ClassJsonField('tension', float)])
    class LugNut:

      def __init__(self, tension: float) -> None:
        self.tension = tension

    @ClassJsonClass([ClassJsonField('number', int),
                     ClassJsonField('lugNuts', [LugNut])])
    class Wheel:

      def __init__(self, number: int, lugNuts: List[LugNut]) -> None:
        self.number = number
        self.lugNuts = lugNuts

    @ClassJsonClass([ClassJsonField('wheel', Wheel),
                     ClassJsonField('wheelList', [Wheel])])
    class Car:

      def __init__(self, wheel: Wheel, wheelList: List[Wheel]) -> None:
        self.wheel = wheel
        self.wheelList = wheelList

    wheelList = []
    for i in range(1, 5):
      wheelList.append(Wheel(i, [LugNut(1), LugNut(2)]))

    expected = '''\
{
  "_class" : "Car",
  "wheel" : <Wheel>,
  "wheelList" : <[Wheel]>
}
  {
    "_class" : "Wheel",
    "number" : <int>,
    "lugNuts" : <[LugNut]>
  }
    {
      "_class" : "LugNut",
      "tension" : <float>
    }'''

    expectedHtml = '''\
<tr>
  <td></td>
  <td>&#123;</td>
</tr>
<tr>
  <td></td>
  <td>"_class" : Car,</td>
<tr>
  <td></td>
  <td>"wheel" : &lt;Wheel&gt;,</td>
</tr>
<tr>
  <td></td>
  <td>"wheelList" : &lt;[Wheel]&gt;</td>
</tr>
<tr>
  <td></td>
  <td>&#125;</td>
</tr>

<tr>
  <td></td>
  <td>&#123;</td>
</tr>
<tr>
  <td></td>
  <td>"_class" : Wheel,</td>
<tr>
  <td></td>
  <td>"number" : &lt;int&gt;,</td>
</tr>
<tr>
  <td></td>
  <td>"lugNuts" : &lt;[LugNut]&gt;</td>
</tr>
<tr>
  <td></td>
  <td>&#125;</td>
</tr>

<tr>
  <td></td>
  <td>&#123;</td>
</tr>
<tr>
  <td></td>
  <td>"_class" : LugNut,</td>
<tr>
  <td></td>
  <td>"tension" : &lt;float&gt;</td>
</tr>
<tr>
  <td></td>
  <td>&#125;</td>
</tr>
'''

    actual = ClassJsonSchemaWriter().writeClass(Car)
    actualHtml = ClassJsonSchemaWriter().writeClass(Car, formatForHtml=True)

    self.assertEqual(expected, actual, msg="Nested Values")
    self.assertEqual(expectedHtml, actualHtml, msg="Html Nested Values")
