#-------------------------------------------------------------------
#  TestClassJsonEncoder.py
#
#  The TestClassJsonEncoder module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the ClassJsonEncoder class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonEncoder
from ai.axe.classJson import ClassJsonException
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonLink
from ai.axe.classJson.jsonTypes.JsonObjRegistry import JsonObjRegistry
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import Dict
from typing import List
import json
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestClassJsonEncoder(AxeSimpleTestCase):
  '''Unit test for the ClassJsonEncoder class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister("TestClass1")
    JsonObjRegistry.unregister("TestClass2")
    JsonObjRegistry.unregister("TestClass3")
    JsonObjRegistry.unregister("TestClass4")
    JsonObjRegistry.unregister("TestClass5")
    JsonObjRegistry.unregister("TestListClass")
    JsonObjRegistry.unregister("NamedAmount")
    JsonObjRegistry.unregister("NamedPoint")
    JsonObjRegistry.unregister("NamedThing")
    JsonObjRegistry.unregister('TestClassMusician')
    JsonObjRegistry.unregister('TestClassInstrument')
    JsonObjRegistry.unregister('TestClassBand')

  #-----------------------------------------------------------------
  def testEncodeObject(self) -> None:
    '''Test simple JSON encoding of a class object.
    '''

    @ClassJsonClass([ClassJsonField('name', str),
                     ClassJsonField('x', int),
                     ClassJsonField('y', int),
                     ClassJsonField('items', [int])])
    class TestClass1:
      def __init__(self, name: str, x: int, y: int, items: List[int]) -> None:
        self.name = name
        self.x = x
        self.y = y
        self.items = items

    obj1 = TestClass1("hello", 3, 7, [3, 4, 5])

    expected = ('{"_class": "TestClass1", "name": "hello", "x": 3, "y": 7, ' +
                '"items": [3, 4, 5]}')
    actual = ClassJsonEncoder().encode(obj1)
    self.assertEqual(expected, actual, 'object')

  #-----------------------------------------------------------------
  def testEncodeNestedObject(self) -> None:
    '''Test simple JSON encoding of nested class objects.
    '''

    @ClassJsonClass([ClassJsonField('name2', str),
                    ClassJsonField('x2', int),
                    ClassJsonField('y2', int)])
    class TestClass3:
      def __init__(self, name2, x2, y2) -> None:
        self.name2 = name2
        self.x2 = x2
        self.y2 = y2

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int),
                    ClassJsonField('obj3', TestClass3)])
    class TestClass2:
      def __init__(self, name: str, x: int, y: int, obj3: TestClass3) -> None:
        self.name = name
        self.x = x
        self.y = y
        self.obj3 = obj3

    obj2 = TestClass3("bye", 21, 22)
    obj1 = TestClass2("hello", 3, 7, obj2)

    expected = ('{"_class": "TestClass2", "name": "hello", "x": 3, "y": 7, ' +
                '"obj3": ' +
                '{"_class": "TestClass3", "name2": "bye", "x2": 21, "y2": 22}}')

    actual = ClassJsonEncoder().encode(obj1)

    self.assertEqual(expected, actual, 'nested objects')

  #-----------------------------------------------------------------
  def testUndecoratedClass(self) -> None:
    '''An undecorated class should throw an exception.
    '''

    try:
      ClassJsonEncoder().encode(ClassJsonEncoder())
      self.fail("Failed to throw undecorated class exception")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testMissingProperty(self) -> None:
    '''A missing property should throw an exception.
    '''

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class TestClass4:
      def __init__(self, name: str, x: int, y: int) -> None:
        self.name = name
        self.x = x
        self.y = y

    obj = TestClass4('name', 33, 34)
    del obj.x

    try:
      ClassJsonEncoder().encode(obj)
      self.fail("Failed to throw undecorated class exception")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testWrongType(self) -> None:
    '''A mis-typed property should throw an exception.
    '''

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class TestClass5:
      def __init__(self, name: str, x: int, y: int) -> None:
        self.name = name
        self.x = x
        self.y = y

    obj = TestClass5('name', 33, 34)

    # Make y a string instead of an int.
    obj.y = 'foo' # type: ignore

    try:
      ClassJsonEncoder().encode(obj)
      self.fail("Failed to throw wrong property type exception")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testDictEncode(self) -> None:
    '''Test JSON encoding of a dictionary.
    '''

    @ClassJsonClass([ClassJsonField('theDict', {int: bool})])
    class TestClass3:
      def __init__(self, theDict: Dict[int, bool]) -> None:
        self.theDict = theDict

    theDict = OrderedDict(((1, True), (2, True), (3, False)))
    obj = TestClass3(theDict)

    encodedDict = '[[1, true], [2, true], [3, false]]'
    expected = ('{"_class": "TestClass3", ' +
                '"theDict": ' + encodedDict + '}')
    actual = ClassJsonEncoder().encode(obj)
    self.assertEqual(expected, actual, 'dict')

  #-----------------------------------------------------------------
  def testNestedDictEncode(self) -> None:
    '''Test JSON encoding of nested dictionaries.
    '''

    @ClassJsonClass([ClassJsonField('theDict', {int: {int: bool}})])
    class TestClass4:
      def __init__(self, theDict: Dict[int, Dict[int, bool]]) -> None:
        self.theDict = theDict

    subDict1 = OrderedDict(((1, True), (2, True), (3, False)))
    subDict2 = OrderedDict(((4, True), (5, True), (6, False)))
    theDict: Dict[int, Dict[int, bool]] = OrderedDict(((100, subDict1),
                                                       (101, subDict2)))
    obj = TestClass4(theDict)

    encodedSubDict1 = '[[1, true], [2, true], [3, false]]'
    encodedSubDict2 = '[[4, true], [5, true], [6, false]]'
    encodedDict = ('[[100, ' + encodedSubDict1 + '], ' +
                   '[101, ' + encodedSubDict2 + ']]')
    expected = ('{"_class": "TestClass4", ' +
                '"theDict": ' + encodedDict + '}')
    actual = ClassJsonEncoder().encode(obj)
    self.assertEqual(expected, actual, 'nested dict')

  #-----------------------------------------------------------------
  def testNestedListEncode(self) -> None:
    '''Test JSON encoding of nested lists.
    '''

    @ClassJsonClass([ClassJsonField('theList', [[[int]]])])
    class TestListClass:
      def __init__(self, theList: List[List[List[int]]]) -> None:
        self.theList = theList

    theList = [
      [[0, 1], [0, 2], [0, 3]],
      [[1, 1], [1, 2], [1, 3]],
      ]

    obj = TestListClass(theList)

    actual = ClassJsonEncoder().encode(obj)

    actualList = json.loads(actual)['theList']

    self.assertEqual([0, 1], actualList[0][0], '0, 0')
    self.assertEqual([1, 1], actualList[1][0], '1, 0')
    self.assertEqual([1, 3], actualList[1][2], '1, 2')

  #-----------------------------------------------------------------
  def testSubclass(self) -> None:
    '''Test JSON decoding of a subclass.
    '''

    # pylint: disable = W0612

    # A base class (NamedThing) and two subclasses (NamedPoint, NamedAmount).

    @ClassJsonClass([], isAbstract=True)
    class NamedThing:
      def __init__(self, name: str) -> None:
        self.name = name

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class NamedPoint(NamedThing):
      def __init__(self, name: str, x: int, y: int) -> None:
        NamedThing.__init__(self, name)
        self.x = x
        self.y = y

    @ClassJsonClass([ClassJsonField('name', str),
                     ClassJsonField('amount', int)])
    class NamedAmount(NamedThing):
      def __init__(self, name: str, amount: int) -> None:
        NamedThing.__init__(self, name)
        self.amount = amount

    obj1 = NamedPoint("hello", 3, 7)
    expected = '{"_class": "NamedPoint", "name": "hello", "x": 3, "y": 7}'

    actual = ClassJsonEncoder().encode(obj1)

    self.assertEqual(expected, actual, 'NamedPoint')

    obj2 = NamedAmount("hello", 42)
    expected = '{"_class": "NamedAmount", "name": "hello", "amount": 42}'

    actual = ClassJsonEncoder().encode(obj2)

    self.assertEqual(expected, actual, 'NamedAmount')

    obj3 = NamedThing("hello")

    try:
      actual = ClassJsonEncoder().encode(obj3)
      self.fail("No exception for encoding abstract class.")

    except ClassJsonException as ex:

      self.assertEqual(True, "abstract class" in str(ex), "abstract message")

  #-----------------------------------------------------------------
  def testLinks(self) -> None:
    '''Test that links are handled correctly.
    '''

    # A band is a list of musicians and instruments.
    # Every musician has an instrument.

    @ClassJsonClass([ClassJsonField('id', int)])
    class TestClassInstrument:
      def __init__(self, id: int) -> None:
        self.id = id

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('instrument',
                                    ClassJsonLink(TestClassInstrument,
                                                  '../instruments',
                                                  ['id']))])
    class TestClassMusician:
      def __init__(self, id: int, instrument: TestClassInstrument) -> None:
        self.id = id
        self.instrument = instrument

    @ClassJsonClass([ClassJsonField('musicians', [TestClassMusician]),
                     ClassJsonField('instruments', [TestClassInstrument])])
    class TestClassBand:
      def __init__(self,
                   musicians: List[TestClassMusician],
                   instruments: List[TestClassInstrument]) -> None:
        self.musicians = musicians
        self.instruments = instruments

    instruments = [
      TestClassInstrument(20),
      TestClassInstrument(21),
      TestClassInstrument(22),
      ]

    musicians = [
      TestClassMusician(10, instruments[2]),
      TestClassMusician(11, instruments[0]),
      ]

    band = TestClassBand(musicians, instruments)


    bandJsonStr = ClassJsonEncoder().encode(band)
    bandJson = json.loads(bandJsonStr)
    musiciansJson: List[Dict[str, object]] = bandJson['musicians']

    musicianJson = musiciansJson[0]
    instrumentJson = musicianJson['instrument']

    self.assertEqual([22], instrumentJson, 'instrument link is the ID.')
