#-------------------------------------------------------------------
#  TestClassJsonDecoder.py
#
#  The TestClassJsonDecoder module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the ClassJsonDecoder class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from .. import ClassJsonClass
from .. import ClassJsonDecoder
from .. import ClassJsonField
from .. import ClassJsonException
from .. import ClassJsonLink
from ..jsonTypes.JsonObjRegistry import JsonObjRegistry
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import List
import json
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestClassJsonDecoder(AxeSimpleTestCase):
  '''Unit test for the ClassJsonDecoder class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister("TestClass1")
    JsonObjRegistry.unregister("TestClass2")
    JsonObjRegistry.unregister("TestClass3")
    JsonObjRegistry.unregister("TestClass4")
    JsonObjRegistry.unregister("TestClass5")
    JsonObjRegistry.unregister("TestListClass")
    JsonObjRegistry.unregister("NamedAmount")
    JsonObjRegistry.unregister("NamedPoint")
    JsonObjRegistry.unregister("NamedThing")
    JsonObjRegistry.unregister('TestClassMusician')
    JsonObjRegistry.unregister('TestClassInstrument')
    JsonObjRegistry.unregister('TestClassBand')

  #-----------------------------------------------------------------
  def testDecodeObject(self) -> None:
    '''Test simple JSON decoding of a class object.
    '''

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)],
                   ctor='create')
    class TestClass1:
      def __init__(self, name, x, y) -> None:
        self.name = name
        self.x = x
        self.y = y
      @staticmethod
      def create() -> 'TestClass1':
        return TestClass1('', 0, 0)

    jsonStr = '{"_class": "TestClass1", "name": "hello", "x": 3, "y": 7}'

    actual = ClassJsonDecoder().decode(jsonStr, TestClass1)

    self.assertEqual('TestClass1', actual.__class__.__name__, 'class name')
    self.assertEqual('hello', actual.name, 'name')
    self.assertEqual(3, actual.x, 'x')
    self.assertEqual(7, actual.y, 'y')

  #-----------------------------------------------------------------
  def testDecodeNestedObject(self) -> None:
    '''Test simple JSON decoding of nested class objects.
    '''

    @ClassJsonClass([ClassJsonField('name2', str),
                    ClassJsonField('x2', int),
                    ClassJsonField('y2', int)])
    class TestClass3:
      def __init__(self) -> None:
        self.name2 = ''
        self.x2 = 0
        self.y2 = 0

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int),
                    ClassJsonField('obj3', TestClass3)],
                   ctor='create')
    class TestClass2:
      def __init__(self, name, x, y, obj3) -> None:
        self.name = name
        self.x = x
        self.y = y
        self.obj3 = obj3
      @staticmethod
      def create() -> 'TestClass2':
        return TestClass2('', 0, 0, None)

    jsonStr = ('{"_class": "TestClass2", "name": "hello", "x": 3, "y": 7, ' +
               '"obj3": ' +
               '{"_class": "TestClass3", "name2": "bye", "x2": 21, "y2": 22}}')

    actual = ClassJsonDecoder().decode(jsonStr, TestClass2)

    self.assertEqual('TestClass2', actual.__class__.__name__, 'class name')
    self.assertEqual('hello', actual.name, 'name')
    self.assertEqual(3, actual.x, 'x')
    self.assertEqual(7, actual.y, 'y')

    obj3 = actual.obj3

    self.assertEqual('TestClass3', obj3.__class__.__name__, 'class name')
    self.assertEqual('bye', obj3.name2, 'name')
    self.assertEqual(21, obj3.x2, 'x')
    self.assertEqual(22, obj3.y2, 'y')

  #-----------------------------------------------------------------
  def testWrongClass(self) -> None:
    '''An unknown class should throw an exception.
    '''

    @ClassJsonClass([])
    class TestClass1:
      def __init__(self) -> None:
        pass

    # pylint: disable=unused-variable
    @ClassJsonClass([])
    class TestClass2:
      def __init__(self) -> None:
        pass
    # pylint: enable=unused-variable

    jsonStr = '{"_class": "TestClass2"}'

    # Decode a TestClass2, but say we're expecting a TestClass1.

    try:
      ClassJsonDecoder().decode(jsonStr, TestClass1)
      self.fail("Failed to throw unknown class exception")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testUnknownClass(self) -> None:
    '''An unknown class should throw an exception.
    '''

    jsonStr = '{"_class": "bad_class_name"}'

    try:
      ClassJsonDecoder().decodeObject(jsonStr)
      self.fail("Failed to throw unknown class exception")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testMissingClass(self) -> None:
    '''A missing _class attribute should throw an exception.
    '''

    jsonStr = '{}'

    try:
      ClassJsonDecoder().decodeObject(jsonStr)
      self.fail("Failed to throw unknown class exception")
    except ClassJsonException as ex:
      msg = str(ex)
      self.assertEqual(True, '_class' in msg, 'correct exception message')

  #-----------------------------------------------------------------
  def testExtraProperty(self) -> None:
    '''An extra property should throw an exception.
    '''

    # pylint: disable = W0612

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class TestClass4:
      def __init__(self) -> None:
        self.name = ''
        self.x = 0
        self.y = 0

    jsonStr = ('{"_class": "TestClass4", "name": "a name", "x": 45, "y": 46,' +
               '"extraProp": "blah"}')

    try:
      ClassJsonDecoder().decode(jsonStr, TestClass4)
      self.fail("Failed to throw extra property exception")
    except ClassJsonException as ex:
      self.assertEqual(True, 'extra properties' in str(ex))

  #-----------------------------------------------------------------
  def testMissingProperty(self) -> None:
    '''A missing property should throw an exception.
    '''

    # pylint: disable = W0612

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class TestClass4:
      def __init__(self) -> None:
        self.name = ''
        self.x = 0
        self.y = 0

    jsonStr = '{"_class": "TestClass4", "name": "a name", "x": 45}'

    try:
      ClassJsonDecoder().decode(jsonStr, TestClass4)
      self.fail("Failed to throw missing property exception")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testBadType(self) -> None:
    '''Test an exception is thrown for a bad field type.
    '''

    # pylint: disable = W0612

    @ClassJsonClass([ClassJsonField('name2', str),
                    ClassJsonField('x2', int),
                    ClassJsonField('y2', int)])
    class TestClass5:
      def __init__(self) -> None:
        self.name2 = ''
        self.x2 = 0
        self.y2 = 0

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int),
                    ClassJsonField('obj3', TestClass5)])
    class TestClass4:
      def __init__(self, name, x, y, obj3) -> None:
        self.name = name
        self.x = x
        self.y = y
        self.obj3 = obj3

    jsonStr = ('{"_class": "TestClass4", "name": "hello", "x": 3, "y": 7, ' +
               '"obj3": ' +
               '{"_class": "TestClass5", "name2": "bye", "x2": 21, ' +
               '"y2": "bad value"}}')

    try:
      ClassJsonDecoder().decode(jsonStr, TestClass4)
      self.fail("Failed to throw a bad field type exception.")
    except ClassJsonException:
      pass

  #-----------------------------------------------------------------
  def testDictDecode(self) -> None:
    '''Test JSON decoding of a dictionary.
    '''

    # pylint: disable = W0612

    @ClassJsonClass([ClassJsonField('theDict', {int: bool})])
    class TestClass3:
      def __init__(self, theDict) -> None:
        self.theDict = theDict

    encodedDict = '[[1, true], [2, true], [3, false]]'
    jsonStr = ('{"_class": "TestClass3", ' +
               '"theDict": ' + encodedDict + '}')

    expectedDict = OrderedDict(((1, True), (2, True), (3, False)))

    actualObj = ClassJsonDecoder().decode(jsonStr, TestClass3)
    self.assertEqual(expectedDict, actualObj.theDict, 'dict')

  #-----------------------------------------------------------------
  def testNestedListDecode(self) -> None:
    '''Test JSON decoding of nested lists.
    '''

    # pylint: disable=W0612
    @ClassJsonClass([ClassJsonField('theList', [[[int]]])])
    class TestListClass:
      def __init__(self, theList) -> None:
        self.theList = theList

    theList = [
      [[0, 1], [0, 2], [0, 3]],
      [[1, 1], [1, 2], [1, 3]],
      ]

    jsonStr = ('{"_class": "TestListClass", ' +
               '"theList":  [ [[0, 1], [0, 2], [0, 3]],' +
               '[[1, 1], [1, 2], [1, 3]]]}')

    actual = ClassJsonDecoder().decode(jsonStr, TestListClass)

    self.assertEqual(theList, actual.theList, 'decoded list')

  #-----------------------------------------------------------------
  def testSubclass(self) -> None:
    '''Test JSON decoding of a subclass.
    '''

    # pylint: disable = W0612

    # A base class (NamedThing) and two subclasses (NamedPoint, NamedAmount).

    @ClassJsonClass([], isAbstract=True)
    class NamedThing:
      def __init__(self, name) -> None:
        self.name = name

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class NamedPoint(NamedThing):
      def __init__(self, name, x, y) -> None:
        NamedThing.__init__(self, name)
        self.x = x
        self.y = y

    @ClassJsonClass([ClassJsonField('name', str),
                     ClassJsonField('amount', int)])
    class NamedAmount(NamedThing):
      def __init__(self, name, amount) -> None:
        NamedThing.__init__(self, name)
        self.amount = amount

    jsonStr = '{"_class": "NamedPoint", "name": "hello", "x": 3, "y": 7}'

    actual1 = ClassJsonDecoder().decode(jsonStr, NamedPoint)

    self.assertEqual('NamedPoint', actual1.__class__.__name__, 'class name')
    self.assertEqual('hello', actual1.name, 'name')
    self.assertEqual(3, actual1.x, 'x')
    self.assertEqual(7, actual1.y, 'y')

    jsonStr = '{"_class": "NamedAmount", "name": "hello", "amount": 42}'

    actual2 = ClassJsonDecoder().decode(jsonStr, NamedAmount)

    self.assertEqual('NamedAmount', actual2.__class__.__name__, 'class name')
    self.assertEqual('hello', actual2.name, 'name')
    self.assertEqual(42, actual2.amount, 'amount')


    jsonStr = '{"_class": "NamedThing", "name": "hello"}'

    try:
      actual = ClassJsonDecoder().decode(jsonStr, NamedThing)
      self.fail("No exception for decoding abstract class.")

    except ClassJsonException as ex:

      self.assertEqual(True, "abstract class" in str(ex), "abstract message")

  #-----------------------------------------------------------------
  def testLinks(self) -> None:
    '''Test that links are handled correctly.
    '''

    # A band is a list of musicians and instruments.
    # Every musician has an instrument.

    @ClassJsonClass([ClassJsonField('id', int)])
    class TestClassInstrument:
      def __init__(self, id: int) -> None:
        self.id = id

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('instrument',
                                    ClassJsonLink(TestClassInstrument,
                                                  '../instruments',
                                                  ['id']))])
    class TestClassMusician:
      def __init__(self, id: int, instrument: TestClassInstrument) -> None:
        self.id = id
        self.instrument = instrument

    @ClassJsonClass([ClassJsonField('musicians', [TestClassMusician]),
                     ClassJsonField('instruments', [TestClassInstrument])])
    class TestClassBand:
      def __init__(self,
                   musicians: List[TestClassMusician],
                   instruments: List[TestClassInstrument]) -> None:
        self.musicians = musicians
        self.instruments = instruments

    instruments = [
      {'_class': 'TestClassInstrument', 'id': 20},
      {'_class': 'TestClassInstrument', 'id': 21},
      {'_class': 'TestClassInstrument', 'id': 22},
      ]

    musicians = [
      {'_class': 'TestClassMusician', 'id': 10, 'instrument': [22]},
      {'_class': 'TestClassMusician', 'id': 11, 'instrument': [20]},
      ]

    bandJson = {'_class': 'TestClassBand',
                'musicians': musicians,
                'instruments': instruments}

    bandJsonStr = json.dumps(bandJson)
    band = ClassJsonDecoder().decode(bandJsonStr, TestClassBand)

    self.assertTrue(band.instruments[2] is band.musicians[0].instrument,
                    'instrument link 0 decoded')
    self.assertTrue(band.instruments[0] is band.musicians[1].instrument,
                    'instrument link 1 decoded')
