#-------------------------------------------------------------------
#  TestClassJsonClass.py
#
#  The TestClassJsonClass module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the ClassJsonClass class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonException
from ai.axe.classJson.jsonTypes.JsonObjRegistry import JsonObjRegistry
from ai.axe.build.unittest import AxeSimpleTestCase
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestClassJsonClass(AxeSimpleTestCase):
  '''Unit test for the ClassJsonClass class.
'''

  # pylint: disable=E1101

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testStaticCtor(self) -> None:
    '''Test using a static method as a constructor.
    '''

    @ClassJsonClass([ClassJsonField('intValue', int)],
                    ctor='create')
    class StaticCtorTestClass:

      def __init__(self) -> None:
        self.intValue = 4

      @staticmethod
      def create(intValue) -> 'StaticCtorTestClass':
        obj = StaticCtorTestClass()
        obj.intValue = intValue
        return obj

    createMethod = StaticCtorTestClass.create
    desc = JsonObjRegistry.getForName('StaticCtorTestClass').desc

    self.assertEqual(createMethod, desc.ctor, 'static ctor')

  #-----------------------------------------------------------------
  def testInvalidStaticCtor(self) -> None:
    '''Test using an invalid static method as a constructor.
    '''

    # pylint: disable=W0612

    try:

      @ClassJsonClass([ClassJsonField('intValue', int)],
                      ctor='badCtorName')
      class StaticCtorTestClass:
        def __init__(self) -> None:
          self.intValue = 4

      self.fail("No exception for bad constructor name.")

    except ClassJsonException as ex:

      isOk = 'badCtorName must be a static method' in str(ex)
      self.assertEqual(True, isOk, 'bad message: %s' % str(ex))

  #-----------------------------------------------------------------
  def testInvalidCtor(self) -> None:
    '''A ctor with the wrong arguments should generate an exception.
    '''

    # pylint: disable=W0612

    # Init method.

    try:

      @ClassJsonClass([ClassJsonField('intValue', int)])
      class BadArgCtorTestClass:
        def __init__(self, fooValue) -> None:
          self.intValue = fooValue

      self.fail("No exception for bad constructor args.")

    except ClassJsonException as ex:

      isOk = 'arguments that are different from constructor' in str(ex)
      self.assertEqual(True, isOk, 'bad message: %s' % str(ex))


    # Static constructor method.

    try:

      @ClassJsonClass([ClassJsonField('intValue', int)],
                      ctor='create')
      class BadArgStaticCtorTestClass:

        def __init__(self) -> None:
          self.intValue: int = 3

        @staticmethod
        def create(fooValue: int) -> 'BadArgStaticCtorTestClass':
          obj = BadArgStaticCtorTestClass()
          obj.intValue = fooValue
          return obj

      self.fail("No exception for bad static constructor args.")

    except ClassJsonException as ex:

      isOk = 'arguments that are different from constructor' in str(ex)
      self.assertEqual(True, isOk, 'bad message: %s' % str(ex))

  #-----------------------------------------------------------------
  def testRedecorating(self):
    '''Test that re-decorating throws an exception.
    '''

    # pylint: disable=C0103
    # pylint: disable=E0102
    # pylint: disable=W0612

    @ClassJsonClass([ClassJsonField('num', int)])
    class A:
      def __init__(self, num) -> None:
        self.num = num

    @ClassJsonClass([ClassJsonField('num', int),
                     ClassJsonField('a', A)])
    class B:
      def __init__(self, num, a) -> None:
        self.num = num
        self.a = a

    # Trying to re-decorate a class throws an exception.

    desc = ClassJsonClass([ClassJsonField('num', int)])

    try:
      desc.decorate(A)

      self.fail("Failed to throw an exception re-decorating a class.")

    except ClassJsonException as ex:

      self.assertEqual(True, 'already been decorated' in str(ex))

    # Trying to decorate two classes with the same name throws an exception.

    desc = ClassJsonClass([ClassJsonField('num', int)])

    try:

      @ClassJsonClass([ClassJsonField('num', int)])
      class A:
        def __init__(self, num) -> None:
          self.num = num

      self.fail("Failed to throw an exception decorating same-named class.")

    except ClassJsonException as ex:

      self.assertEqual(True, 'already been decorated' in str(ex))

    JsonObjRegistry.unregister("A")
    JsonObjRegistry.unregister("B")
