#-------------------------------------------------------------------
#  TestJsonPathElement.py
#
#  The TestJsonPathElement module.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the JsonPathElement class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.classJson.link import JsonPathElement
from ai.axe.classJson.link import JsonPathParseException
from ai.axe.classJson.link import JsonPathResolveException
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonPathElement(AxeSimpleTestCase):
  '''Unit test for the JsonPathElement class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testCtor(self) -> None:
    '''Test creating the object.
    '''

    element = JsonPathElement("foo", 3)

    self.assertEqual("foo", element.name, "name")
    self.assertEqual(3, element.index, "index")

    # Check that no exception raised and raise coverage.
    str(element)

  #-----------------------------------------------------------------
  def testParse(self) -> None:
    '''Test parsing strings.
    '''

    element = JsonPathElement.parse("foo[3]", "path/to/foo[3]")

    self.assertEqual("foo", element.name, "name")
    self.assertEqual(3, element.index, "index")

    element = JsonPathElement.parse("foo", "path/to/foo")

    self.assertEqual("foo", element.name, "name")
    self.assertEqual(None, element.index, "index")


    # Unbalanced brackets.

    try:
      JsonPathElement.parse("foo[3]]", "path/to/foo[3]]")
      self.fail("No exception for unbalanced brackets.")

    except JsonPathParseException:
      pass

    # Too many brackets.

    try:
      JsonPathElement.parse("foo[3][2]", "path/to/foo[3][2]")
      self.fail("No exception for too many brackets.")

    except JsonPathParseException:
      pass

    # Reversed brackets.

    try:
      JsonPathElement.parse("foo]3[", "path/to/foo]3[")
      self.fail("No exception for reversed brackets.")

    except JsonPathParseException:
      pass

    # Bad index.

    try:
      JsonPathElement.parse("foo[badIndex]", "path/to/foo[badIndex]")
      self.fail("No exception for bad index.")

    except JsonPathParseException:
      pass

  #-----------------------------------------------------------------
  def testResolve(self) -> None:
    '''Test resolving.
    '''

    class Child:
      def __init__(self) -> None:
        pass

    class Parent:
      def __init__(self, theChild: Child, children: List[Child]) -> None:
        self.theChild = theChild
        self.children = children

    child = Child()
    children = [Child(), Child(), Child()]
    parent = Parent(child, children)

    element = JsonPathElement.parse("theChild", "theChild")
    resolved = element.resolve(parent)

    self.assertEqual(child, resolved, "no index")

    element = JsonPathElement.parse("children[1]", "children[1]")
    resolved = element.resolve(parent)

    self.assertEqual(children[1], resolved, "index 1")

    # Child is not a list.

    element = JsonPathElement.parse("theChild[1]", "theChild[1]")
    try:
      element.resolve(parent)
      self.fail("No exception for not a list.")

    except JsonPathResolveException:
      pass

    # List too short.

    element = JsonPathElement.parse("children[4]", "children[4]")
    try:
      element.resolve(parent)
      self.fail("No exception for list too short.")

    except JsonPathResolveException:
      pass
