#-------------------------------------------------------------------
#  TestJsonPath.py
#
#  The TestJsonPath module.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the JsonPath class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.classJson.link import JsonPath
from ai.axe.classJson.link import JsonPathParseException
from ai.axe.classJson.link import JsonPathResolveException
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonPath(AxeSimpleTestCase):
  '''Unit test for the JsonPath class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testParsing(self) -> None:
    '''Test the parse() static constructor.
    '''

    path = JsonPath.parse('../a/b[42]/c')

    self.assertEqual('..', path.elements[0].name, 'elements 0')
    self.assertEqual('a', path.elements[1].name, 'elements 1')
    self.assertEqual('b', path.elements[2].name, 'elements 2')
    self.assertEqual('c', path.elements[3].name, 'elements 3')

    self.assertEqual(None, path.elements[0].index, 'elements 0')
    self.assertEqual(None, path.elements[1].index, 'elements 1')
    self.assertEqual(42, path.elements[2].index, 'elements 2')
    self.assertEqual(None, path.elements[3].index, 'elements 3')

    # Illegal input is rejected.

    try:
      JsonPath.parse('/../a/b/c')
      self.fail("No exception for a leading slash.")

    except JsonPathParseException:
      pass

    # String conversion.

    # Don't check format, just make sure no exception is thrown.
    str(path)

    self.assertEqual('../a/b[42]/c', path.toString(), 'toString')

  #-----------------------------------------------------------------
  def testResolving(self) -> None:
    '''Test resolving a path.
    '''

    # Allow short class and variable names.
    # pylint: disable=invalid-name

    # A tree of objects for resolving.

    class Root:
      def __init__(self):
        self.a = A()

    class A:
      def __init__(self):
        self.b1 = B1()
        self.b2 = B2()

    class B1:
      def __init__(self):
        self.c1a = C1a()
        self.c1b = C1b()

    class B2:
      def __init__(self):
        self.c2a = C2a()
        self.c2b = C2b()

    class C1a:
      pass
    class C1b:
      pass
    class C2a:
      pass
    class C2b:
      pass

    root = Root()

    # Resolve from the top down.

    expected = root.a.b2.c2a
    actual = JsonPath.parse('a/b2/c2a').resolve([], root)

    self.assertEqual(expected, actual, 'a/b2/c2a')

    # Resolve from the bottom up.

    parents = [root,
               root.a,
               root.a.b2]

    expected = root
    actual = JsonPath.parse('../../..').resolve(parents, root.a.b2.c2a)

    self.assertEqual(expected, actual, '../../..')

    # Go up and down to find a sibling.

    parents = [root,
               root.a,
               root.a.b2]

    expected = root.a.b1.c1b
    actual = JsonPath.parse('../../../a/b1/c1b').resolve(parents, root.a.b2.c2a)

    self.assertEqual(expected, actual, '../../../a/b1/c1b')

    # An empty path resolves to the start object.

    expected = root
    actual = JsonPath([]).resolve([], root)

    self.assertEqual(expected, actual, 'empty path')

    # Try to go up past the root.

    parents = [root,
               root.a,
               root.a.b2]

    try:
      JsonPath.parse('../../../..').resolve(parents, root.a.b2.c2a)
      self.fail("No exception going up past the root.")

    except JsonPathResolveException as ex:

      self.assertTrue('no parents' in str(ex.__cause__),
                      'going up past the root')
